/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Singleton class to handle export action to LHEO.
 * @private
 */
Ext.define('Ametys.plugins.odf.lheo.ExportLHEO', {
	singleton: true,

 	/**
	 * Exports to LHEO
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
	 */
	exportToLHEO: function (controller)
	{
        var target = controller.getMatchingTargets()[0];
        
        var params = { id: target.getParameters().id };
        Ext.Object.each(Ametys.getAppParameters(), function(key, value) {
		    if (Ext.isString(value)) 
            {
		        params[key] = value;
		    }
		});

        Ametys.openWindow(Ametys.getPluginDirectPrefix('odf') + '/export-lheo.xml', params);
	},
	
	/**
     * Exports all programs from catalog to LHEO
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     */
    exportAllToLHEO: function (controller)
    {
        this._delayedInitialize();
        this._box.show();
        this._initForm();
    },
    
        /**
     * @private
     * Creates the dialog box
     */
    _delayedInitialize: function ()
    {
        if (!this._initialized)
        {
            this._formPanel = this._createFormPanel();
            
            this._box = Ext.create('Ametys.window.DialogBox', {
                title: "{{i18n PLUGINS_ODF_EXPORT_GLOBAL_LHEO_LABEL}}",
                iconCls: 'odficon-blackboard',
                
                width: 450,
                scrollable: false,
                
                items: [this._formPanel],
                
                defaultFocus: 'title',
                closeAction: 'hide',
                buttons: [{
                    text: "{{i18n PLUGINS_ODF_EXPORT_GLOBAL_LHEO_OK_BUTTON}}",
                    handler: Ext.bind(this._validate, this)
                }, {
                    text: "{{i18n PLUGINS_ODF_EXPORT_GLOBAL_LHEO_CANCEL_BUTTON}}",
                    handler: Ext.bind( function() {this._box.close();}, this)
                }]    
            });
            
            this._initialized = true;
        }
    },
    
    /**
     * Creates the form panel of this dialog box.
     * @return {Ext.form.Panel} The form panel
     * @private
     */
    _createFormPanel: function()
    {
        var formPanel = Ext.create('Ext.form.Panel', {
            defaultType: 'combo',
            defaults: {
                cls: 'ametys',
                labelSeparator: '',
                labelAlign: 'right',
                labelWidth: 80,
                width: '100%',
                msgTarget: 'side'
            },
            
            border: false,
            scrollable: true,
            
            items: [
                {
                    xtype: 'container',
                    itemId: 'hint',
                    html: "{{i18n PLUGINS_ODF_EXPORT_GLOBAL_LHEO_CHOOSE_CATALOG_HINT}}",
                    cls: 'a-text'
                },
                Ametys.odf.catalog.CatalogDAO.createComboBox({
                    name: 'catalog',
                    itemId: 'catalog',
                    allowBlank: false
                })
            ]
            
        });
        
        return formPanel;
    },
    
    /**
     * @private
     * Initializes the form with some optional values.
     */
    _initForm: function ()
    {
        var form = this._formPanel.getForm();
        form.reset();
        
        Ametys.odf.catalog.CatalogDAO.getDefaultCatalogName([], function(catalogName) {
            form.findField('catalog').setValue(catalogName)
        }, this);
    },
    
    /**
     * @private
     * Handler for the 'ok' button of the dialog box
     */ 
    _validate: function ()
    {
        var form = this._formPanel.getForm();
        if (!form.isValid())
        {
            return;
        }
        
        var values = form.getValues();
        values.lang = Ametys.cms.language.LanguageDAO.getCurrentLanguage();
        
        Ametys.openWindow(Ametys.getPluginDirectPrefix('odf') + '/export-global-lheo.xml', values);
        
        this._box.close();
    }
});