/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.catalog;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.ametys.cms.repository.Content;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.data.type.EducationalPathRepositoryElementType;
import org.ametys.plugins.repository.data.holder.impl.DataHolderHelper;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Copy updater to update the educational path values of courses after a catalog copy
 */
public class EducationalPathCopyUpdater extends AbstractLogEnabled implements CopyCatalogUpdater
{
    @Override
    // Be careful ! For now this method only update educational paths of courses
    public void updateContents(String initialCatalogName, String newCatalogName, Map<Content, Content> copiedContents, Content targetParentContent)
    {
        for (Content copiedContent : copiedContents.values())
        {
            if (copiedContent instanceof Course course)
            {
                Map<String, Object> educationalPathAttributes = DataHolderHelper.findEditableItemsByType(course, EducationalPathRepositoryElementType.EDUCATIONAL_PATH_ELEMENT_TYPE_ID);
                for (Map.Entry<String, Object> entry : educationalPathAttributes.entrySet())
                {
                    String dataPath = entry.getKey();
                    Object value = entry.getValue();
                    
                    if (value instanceof EducationalPath path)
                    {
                        EducationalPath copiedPath = _getUpdatedEducationalPath(course, path, copiedContents);
                        course.setValue(dataPath, copiedPath);
                    }
                    else if (value instanceof EducationalPath[] paths)
                    {
                        List<EducationalPath> newPaths = new ArrayList<>();
                        for (EducationalPath path : paths)
                        {
                            EducationalPath copiedPath = _getUpdatedEducationalPath(course, path, copiedContents);
                            if (copiedPath == null)
                            {
                                newPaths.add(copiedPath);
                            }
                        }
                        
                        course.setValue(dataPath, newPaths.toArray(EducationalPath[]::new));
                    }
                }
                
                if (course.needsSave())
                {
                    course.saveChanges();
                }
            }
        }
    }
    
    private EducationalPath _getUpdatedEducationalPath(Content content, EducationalPath initialPath, Map<Content, Content> copiedContents)
    {
        List<ProgramItem> programItems = new ArrayList<>();
        for (String programItemId : initialPath.getProgramItemIds())
        {
            ProgramItem copiedProgramItem = copiedContents.entrySet()
                                                          .stream()
                                                          .filter(entry -> programItemId.equals(entry.getKey().getId()))
                                                          .map(Map.Entry::getValue)
                                                          .filter(ProgramItem.class::isInstance)
                                                          .findFirst()
                                                          .map(ProgramItem.class::cast)
                                                          .orElse(null);
            if (copiedProgramItem == null)
            {
                getLogger().warn("No corresponding content found with id '{}' in copied catalog. Unable to update the educational path value for content '{}'", programItemId, content.getId());
                return null;
            }
            
            programItems.add(copiedProgramItem);
        }
        
        return EducationalPath.of(programItems.toArray(ProgramItem[]::new));
    }
}
