/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.clientsideelement;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.repository.ModifiableContent;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.odf.cdmfr.CDMFRHandler;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * Display the button only if the 'odf.publish.server' parameter is checked.
 */
public class RepublishContentClientSideElement extends StaticClientSideElement
{
    /** The CDM-fr handler */
    protected CDMFRHandler _cdmfrHandler;
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _cdmfrHandler = (CDMFRHandler) smanager.lookup(CDMFRHandler.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public List<Script> getScripts(boolean ignoreRights, Map<String, Object> contextParameters)
    {
        if (_cdmfrHandler.needToHandle())
        {
            return super.getScripts(ignoreRights, contextParameters);
        }
        
        return new ArrayList<>();
    }
    
    /**
     * Publish CDM-fr for the given contents
     * @param contentIds the content ids
     * @return the result
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> publish(List<String> contentIds)
    {
        Map<String, Object> results = new HashMap<>();
        
        try
        {
            List<ModifiableContent> contents = contentIds.stream()
                    .map(this::_getContent)
                    .toList();
            
            _cdmfrHandler.handleCDMFR(contents);
        }
        catch (IllegalAccessError e)
        {
            results.put("error", "no-right");
        }
        catch (Exception e)
        {
            results.put("error", "error");
        }
        
        return results;
    }
    
    private ModifiableContent _getContent(String contentId)
    {
        ModifiableContent content = _resolver.resolveById(contentId);
        if (_rightManager.currentUserHasRight("ODF_Rights_Program_Validate", content) != RightResult.RIGHT_ALLOW)
        {
            throw new IllegalAccessError("Cannot republish content with id '" + content.getId() + "' without convenient rights");
        }
        
        return content;
    }
    
}
