/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.content;

import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.Enumerator;

/**
 * {@link Enumerator} for previous/next available years of a year container
 */
public class YearsEnumerator implements Enumerator<ContentValue>, Serviceable, Contextualizable
{
    private Context _context;
    private ODFHelper _odfHelper;

    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
    }
    
    // FIXME CMS-12608: it should be not be necessary to override this method. Temporary fix.
    public I18nizableText getEntry(ContentValue value) throws Exception
    {
        try
        {
            return _getDisplayName((Container) value.getContent());
        }
        catch (UnknownAmetysObjectException e)
        {
            // Ignore.
        }
        
        return null;
    }

    public Map<ContentValue, I18nizableText> getEntries() throws Exception
    {
        Optional<Container> container = Optional.of(_context)
                                                .map(ContextHelper::getRequest)
                                                .map(r -> r.getAttribute(Content.class.getName()))
                                                .filter(Container.class::isInstance)
                                                .map(Container.class::cast);
        
        if (container.isEmpty())
        {
            // Not in a container context
            return Map.of();
        }
        
        Container containerObj = container.get();
        if (!_odfHelper.isContainerOfTypeYear(containerObj))
        {
            // Not a year container
            return Map.of();
        }
        
        Set<Program> parentPrograms = _odfHelper.getParentPrograms(containerObj);
        
        Set<Container> years = new HashSet<>();
        for (Program program : parentPrograms)
        {
            years.addAll(_odfHelper.getYears(program));
        }
        
        // Remove the current year
        years.remove(containerObj);
        
        return years.stream().collect(Collectors.toMap(ContentValue::new, this::_getDisplayName));
    }
    
    private I18nizableText _getDisplayName(Container container)
    {
        StringBuilder displayName = new StringBuilder();
        
        // Display code
        displayName.append("[");
        displayName.append(container.getDisplayCode());
        displayName.append("] ");
        
        displayName.append(container.getTitle());
        
        return new I18nizableText(displayName.toString());
    }
}
