/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.content.code;

import java.io.IOException;
import java.net.MalformedURLException;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.Enumerator;

/**
 * An enumeration that list the providers for program items' display code
 */
public class DisplayCodeProviderEnumerator implements Enumerator<String>, Serviceable
{
    /** The display code extension point */
    private DisplayCodeProviderExtensionPoint _displayCodeProviderEP;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _displayCodeProviderEP = (DisplayCodeProviderExtensionPoint) manager.lookup(DisplayCodeProviderExtensionPoint.ROLE);
    }
    
    public Map<String, I18nizableText> getEntries() throws MalformedURLException, IOException
    {
        return _displayCodeProviderEP.getExtensionsIds()
            .stream()
            .map(_displayCodeProviderEP::getExtension)
            .filter(DisplayCodeProvider::isActive)
            .collect(Collectors.toMap(DisplayCodeProvider::getId, DisplayCodeProvider::getLabel));
    }

    public I18nizableText getEntry(String value) throws Exception
    {
        DisplayCodeProvider displayCode = _displayCodeProviderEP.getExtension(value);
        return displayCode != null ? displayCode.getLabel() : null;
    }
}
