/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.content.code;

import java.util.HashMap;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.helper.AmetysIdentifiers;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.core.util.LambdaUtils;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.CourseFactory;
import org.ametys.odf.courselist.CourseListFactory;
import org.ametys.odf.coursepart.CoursePartFactory;
import org.ametys.odf.orgunit.OrgUnitFactory;
import org.ametys.odf.program.ContainerFactory;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.SubProgramFactory;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;

/**
 * Implementation of {@link UniqueCodeGenerator} that generates a code composed by a prefix representing the type of content and an incremental number on 6 digits. 
 */
public class PrefixedIncrementalCodeGenerator extends AbstractUniqueCodeGenerator implements Serviceable
{
    private static final String __DEFAULT_PREFIX_FOR_PROGRAM = "FOR";
    private static final String __DEFAULT_PREFIX_FOR_SUBPROGRAM = "PAR";
    private static final String __DEFAULT_PREFIX_FOR_CONTAINER = "CTN";
    private static final String __DEFAULT_PREFIX_FOR_COURSELIST = "LST";
    private static final String __DEFAULT_PREFIX_FOR_COURSE = "ELP";
    private static final String __DEFAULT_PREFIX_FOR_ORGUNIT = "CMP";
    private static final String __DEFAULT_PREFIX_FOR_COURSEPART = "HE";
    
    private Map<String, String> _prefixByTypes = new HashMap<>();
    
    private ServiceManager _smanager;
    private AmetysObjectResolver _resolver;
    private AmetysIdentifiers _ametysIdentifiers;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _smanager = manager;
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        super.configure(configuration);
        
        // Get configured prefixes
        Map<String, String> configuredPrefixes = Stream.of(configuration.getChildren("prefix"))
                                                    .collect(Collectors.toMap(LambdaUtils.wrap(c -> c.getAttribute("type")), LambdaUtils.wrap(c -> c.getValue())));
        
        
        // Configured prefixes by types with default values
        _prefixByTypes.put(ProgramFactory.PROGRAM_CONTENT_TYPE, configuredPrefixes.getOrDefault(ProgramFactory.PROGRAM_CONTENT_TYPE, __DEFAULT_PREFIX_FOR_PROGRAM));
        _prefixByTypes.put(SubProgramFactory.SUBPROGRAM_CONTENT_TYPE, configuredPrefixes.getOrDefault(SubProgramFactory.SUBPROGRAM_CONTENT_TYPE, __DEFAULT_PREFIX_FOR_SUBPROGRAM));
        _prefixByTypes.put(ContainerFactory.CONTAINER_CONTENT_TYPE, configuredPrefixes.getOrDefault(ContainerFactory.CONTAINER_CONTENT_TYPE, __DEFAULT_PREFIX_FOR_CONTAINER));
        _prefixByTypes.put(CourseListFactory.COURSE_LIST_CONTENT_TYPE, configuredPrefixes.getOrDefault(CourseListFactory.COURSE_LIST_CONTENT_TYPE, __DEFAULT_PREFIX_FOR_COURSELIST));
        _prefixByTypes.put(CourseFactory.COURSE_CONTENT_TYPE, configuredPrefixes.getOrDefault(CourseFactory.COURSE_CONTENT_TYPE, __DEFAULT_PREFIX_FOR_COURSE));
        _prefixByTypes.put(OrgUnitFactory.ORGUNIT_CONTENT_TYPE, configuredPrefixes.getOrDefault(OrgUnitFactory.ORGUNIT_CONTENT_TYPE, __DEFAULT_PREFIX_FOR_ORGUNIT));
        _prefixByTypes.put(CoursePartFactory.COURSE_PART_CONTENT_TYPE, configuredPrefixes.getOrDefault(CoursePartFactory.COURSE_PART_CONTENT_TYPE, __DEFAULT_PREFIX_FOR_COURSEPART));
    }
    
    private AmetysIdentifiers _getAmetysIdentifiers()
    {
        // Lazing loading to keep component safe
        if (_ametysIdentifiers == null)
        {
            try
            {
                _ametysIdentifiers = (AmetysIdentifiers) _smanager.lookup(AmetysIdentifiers.ROLE);
            }
            catch (ServiceException e)
            {
                throw new RuntimeException("Unable to lookup after the ametys identifiers", e);
            }
        }
        
        return _ametysIdentifiers;
    }
    
    private AmetysObjectResolver _getResolver()
    {
        // Lazing loading to keep component safe
        if (_resolver == null)
        {
            try
            {
                _resolver = (AmetysObjectResolver) _smanager.lookup(AmetysObjectResolver.ROLE);
            }
            catch (ServiceException e)
            {
                throw new RuntimeException("Unable to lookup after the ametys object resolver", e);
            }
        }
        
        return _resolver;
    }
    
    public String generateUniqueCode(String cTypeId)
    {
        String prefix = _prefixByTypes.get(cTypeId);
        if (prefix == null)
        {
            throw new IllegalArgumentException(cTypeId + " is not supported to generate ODF content unique code");
        }
        
        long counter = _getAmetysIdentifiers().readCounter(_pluginName, cTypeId) + 1;
        String uniqueCode = prefix + counter;
        while (!_checkUnicity(cTypeId, uniqueCode))
        {
            counter++;
            uniqueCode = prefix + counter;
        }
        _getAmetysIdentifiers().saveCounter(_pluginName, cTypeId, counter);
        
        return uniqueCode;
    }
    
    private boolean _checkUnicity(String cTypeId, String code)
    {
        // Check unicity whatever the language and catalog
        Expression contentTypeExpr = new ContentTypeExpression(Operator.EQ, cTypeId);
        Expression codeExpr = new StringExpression(ProgramItem.CODE, Operator.EQ, code);
        
        Expression expr = new AndExpression(contentTypeExpr, codeExpr);
        
        String xpathQuery = ContentQueryHelper.getContentXPathQuery(expr);
        return _getResolver().query(xpathQuery).getSize() == 0;
    }
}
