/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.course;

import java.time.LocalDate;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.jcr.Node;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.ContentDataHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.RichText;
import org.ametys.cms.data.holder.group.ModifiableIndexableRepeater;
import org.ametys.cms.data.holder.group.ModifiableIndexableRepeaterEntry;
import org.ametys.cms.repository.ModifiableDefaultContent;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.content.code.DisplayCodeProperty;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.courselist.CourseListContainer;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.data.EducationalPath;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeaterEntry;
import org.ametys.runtime.model.exception.UndefinedItemPathException;

/**
 * Class representing a {@link Course}
 */
public class Course extends ModifiableDefaultContent<CourseFactory> implements CourseListContainer, ProgramItem
{
    /** Name of attribute for parent course lists */
    public static final String PARENT_COURSE_LISTS = "parentCourseLists";
    
    /** Name of attribute for parent course lists */
    public static final String CHILD_COURSE_LISTS = "courseLists";
    
    /** Mandatory Identifier to generate the CDM-fr id */
    public static final String CDM_CODE = "cdmCode";
    
    /** Constants for ects attribute */
    public static final String ECTS = "ects";
    
    /** Constants for path of ects attribute by education path */
    public static final String ECTS_BY_PATH = "ectsByEducationalPath/ects";

    /** Constants for level attribute */
    public static final String LEVEL = "level";

    /** Constants for description attribute */
    public static final String DESCRIPTION = "description";

    /** Constants for objectives attribute */
    public static final String OBJECTIVES = "objectives";

    /** Constants for nbHours attribute */
    public static final String NUMBER_OF_HOURS = "nbHours";

    /** Constants for neededPrerequisite attribute */
    public static final String NEEDED_PREREQUISITE = "neededPrerequisite";
    
    /** Constants for teds attribute */
    public static final String TEDS = "teds";

    /** Constants for formOfAssessment attribute */
    public static final String FORM_OF_ASSESSMENT = "formOfAssessment";
    
    /** Constants for repeater 'acquiredMicroSkills' activated with the configuration parameter 'odf.skills.enabled' */
    public static final String ACQUIRED_MICRO_SKILLS = "acquiredMicroSkills";
    /** The attribute name of the program in acquiredMicroSkills activated with the configuration parameter 'odf.skills.enabled' */
    public static final String ACQUIRED_MICRO_SKILLS_PROGRAM = "program";
    /** The attribute name of the microSkills in acquiredMicroSkills activated with the configuration parameter 'odf.skills.enabled' */
    public static final String ACQUIRED_MICRO_SKILLS_SKILLS = "microSkills";
    
    /** Constants for syllabus attribute */
    public static final String SYLLABUS = "syllabus";

    /** Constants for additionalInformations attribute */
    public static final String ADDITIONAL_INFORMATIONS = "additionalInformations";

    /** Constants for erasmusCode attribute */
    public static final String ERASMUS_CODE = "erasmusCode";

    /** Constants for teachingLocation attribute */
    public static final String TEACHING_LOCATION = "teachingLocation";

    /** Constants for maxNumberOfStudents attribute */
    public static final String MAX_NUMBER_OF_STUDENTS = "maxNumberOfStudents";

    /** Constants for teachingTerm attribute */
    public static final String TEACHING_TERM = "teachingTerm";

    /** Constants for timeSlot attribute */
    public static final String TIME_SLOT = "timeSlot";

    /** Constants for trainingCourseDuration attribute */
    public static final String TRAINING_COURSE_DURATION = "trainingCourseDuration";

    /** Constants for teachingMethod attribute */
    public static final String FORMODFTEACHING_METHOD = "formofteachingMethod";
    
    /** Constants for formofteachingOrg attribute */
    public static final String FORMOFTEACHING_ORG = "formofteachingOrg";

    /** Constants for formOfTeaching attribute */
    public static final String TEACHING_ACTIVITY = "teachingActivity";

    /** Constants for teachingLanguage attribute */
    public static final String TEACHING_LANGUAGE = "teachingLanguage";

    /** Constants for startDate attribute */
    public static final String START_DATE = "startDate";

    /** Constants for keywords attribute */
    public static final String KEYWORDS = "keywords";

    /** Constants for webLinkLabel attribute */
    public static final String WEB_LINK_LABEL = "webLinkLabel";

    /** Constants for webLinkUrl attribute */
    public static final String WEB_LINK_URL = "webLinkUrl";

    /** Constants for lomSheets attribute */
    public static final String LOM_SHEETS = "lomSheets";

    /** Constants for LOM Sheet URL attribute */
    public static final String LOM_SHEET_URL = "linkUrl";
    
    /** Constants for LOM Sheet label attribute */
    public static final String LOM_SHEET_LABEL = "linkLabel";
    
    /** Constants for attribute 'contacts' */
    public static final String CONTACTS = "contacts";
    
    /** Constants for attribute 'contacts/role' */
    public static final String CONTACTS_ROLE = "role";
    
    /** Constants for attribute 'contacts/persons' */
    public static final String CONTACTS_PERSONS = "persons";
    
    /** Constants for attribute 'courseType' */
    public static final String COURSE_TYPE = "courseType";
    
    /** Constants for attribute 'bibliography' */
    public static final String BIBLIOGRAPHY = "bibliography";

    /** Constants for attribute 'skills' */
    public static final String SKILLS = "skills";

    /** Constants for attribute 'openToExchangeStudents' */
    public static final String OPEN_TO_EXCHANGE_STUDENTS = "openToExchangeStudents";
    
    /** Constants for attribute 'courseParts' */
    public static final String CHILD_COURSE_PARTS = "courseParts";

    private String _contextPath;

    private List<EducationalPath> _currentEducationalPath;

    /**
     * Constructor.
     * @param node the JCR Node.
     * @param parentPath the parent path
     * @param factory the corresponding factory.
     */
    public Course(Node node, String parentPath, CourseFactory factory)
    {
        super(node, parentPath, factory);
    }

    @Override
    public List<CourseList> getCourseLists()
    {
        return Arrays.stream(getValue(CHILD_COURSE_LISTS, false, new ContentValue[0]))
                .map(ContentValue::getContentIfExists)
                .flatMap(Optional::stream)
                .map(CourseList.class::cast)
                .collect(Collectors.toList());
    }
    
    /**
     * Get the parent course lists
     * @return The parent course lists
     */
    public List<CourseList> getParentCourseLists()
    {
        return Arrays.stream(getValue(PARENT_COURSE_LISTS, false, new ContentValue[0]))
                .map(ContentValue::getContentIfExists)
                .flatMap(Optional::stream)
                .map(CourseList.class::cast)
                .collect(Collectors.toList());
    }

    @Override
    public boolean hasCourseLists()
    {
        return !ContentDataHelper.isMultipleContentDataEmpty(this, CHILD_COURSE_LISTS);
    }
    
    @Override
    public boolean containsCourseList(String clId)
    {
        return ArrayUtils.contains(ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, CHILD_COURSE_LISTS), clId);
    }
    
    // --------------------------------------------------------------------------------------//
    // CONTACTS
    // --------------------------------------------------------------------------------------//

    /**
     * Return the list of Persons in charge binded to this program
     * @return the id of contacts
     */
    public Set<String> getContacts()
    {
        Set<String> contactIds = new HashSet<>();
        
        ModelAwareRepeater contacts = getRepeater(CONTACTS);
        if (contacts != null)
        {
            for (ModelAwareRepeaterEntry contactEntry : contacts.getEntries())
            {
                // Remove empty values
                contactIds.addAll(ContentDataHelper.getContentIdsStreamFromMultipleContentData(contactEntry, CONTACTS_PERSONS)
                        .filter(contentId -> !contentId.isEmpty())
                        .collect(Collectors.toSet()));
            }
        }

        return Collections.EMPTY_SET;
    }
    
    /**
     * Return the list of Persons in charge binded to this program
     * @return a list of roles and UUID
     */
    public Map<String, List<String>> getContactsByRole()
    {
        Map<String, List<String>> contactsByRole = new HashMap<>();
        
        ModelAwareRepeater contacts = getRepeater(CONTACTS);
        if (contacts != null)
        {
            for (ModelAwareRepeaterEntry contactEntry : contacts.getEntries())
            {
                // Remove empty values
                List<String> persons = ContentDataHelper.getContentIdsStreamFromMultipleContentData(contactEntry, CONTACTS_PERSONS)
                                                    .filter(StringUtils::isNotEmpty)
                                                    .collect(Collectors.toList());
                if (!persons.isEmpty())
                {
                    String role = ContentDataHelper.getContentIdFromContentData(contactEntry, CONTACTS_ROLE);
                    contactsByRole.put(role, persons);
                }
            }
        }

        return contactsByRole;
    }
   
    
    // --------------------------------------------------------------------------------------//
    // CONTEXT
    // --------------------------------------------------------------------------------------//
    
    // Méthodes utilisées lors du parcours d'une maquette uniquement, afin de contextualiser l'élément
    // A ne pas utiliser n'importe ou ni n'importe comment
    
    /**
     * Set the parent path for links and breadcrumb
     * @param path the parent path
     */
    public void setContextPath (String path)
    {
        _contextPath = path;
    }
    
    /**
     * Get the parent path. Can be null.
     * @return the parent path
     */
    public String getContextPath ()
    {
        return _contextPath;
    }
    
    /**
     * Set the current educational paths of this course
     * @param paths the current educational paths
     */
    public void setCurrentEducationalPaths(List<EducationalPath> paths)
    {
        _currentEducationalPath = paths;
    }
    
    /**
     * Get the current educational paths of this course
     * @return the current educational path
     */
    public List<EducationalPath> getCurrentEducationalPaths()
    {
        return _currentEducationalPath;
    }
    
    
    // --------------------------------------------------------------------------------------//
    // GETTERS AND SETTERS
    // --------------------------------------------------------------------------------------//
    @Override
    public String getCatalog()
    {
        return getValue(CATALOG);
    }
    
    @Override
    public void setCatalog(String catalog) throws AmetysRepositoryException
    {
        setValue(CATALOG, catalog);
    }
    
    @Override
    public String getCode()
    {
        return getValue(CODE, false, StringUtils.EMPTY);
    }
    
    @Override
    public void setCode(String code) throws AmetysRepositoryException
    {
        setValue(CODE, code);
    }
    
    public String getDisplayCode()
    {
        return getValue(DisplayCodeProperty.PROPERTY_NAME, false, StringUtils.EMPTY);
    }
    
    public boolean isPublishable()
    {
        return getInternalDataHolder().getValue(PUBLISHABLE, true);
    }
    
    public void setPublishable(boolean isPublishable)
    {
        getInternalDataHolder().setValue(PUBLISHABLE, isPublishable);
    }
    
    public List<String> getOrgUnits()
    {
        try
        {
            return ContentDataHelper.getContentIdsListFromMultipleContentData(this, ORG_UNITS_REFERENCES);
        }
        catch (UndefinedItemPathException e)
        {
            return Collections.EMPTY_LIST; // this attribute is not part of model
        }
    }
    
    /**
     * Get the description
     * @return the description or null
     */
    public RichText getDescription()
    {
        return getValue(DESCRIPTION);
    }

    /**
     * Get the objectives
     * @return objectives
     */
    public RichText getObjectives()
    {
        return getValue(OBJECTIVES);
    }
    
    /**
     * Get the course parts.
     * @return The {@link List} of attached {@link CoursePart}s
     */
    public List<CoursePart> getCourseParts()
    {
        return Arrays.stream(getValue(CHILD_COURSE_PARTS, false, new ContentValue[0]))
                .map(ContentValue::getContentIfExists)
                .flatMap(Optional::stream)
                .map(CoursePart.class::cast)
                .collect(Collectors.toList());
    }
    
    /**
     * Get the needed prerequisites
     * @return the needed prerequisites or null if not set
     */
    public RichText getNeededPrerequisite()
    {
        return getValue(NEEDED_PREREQUISITE);
    }

    /**
     * Get the formOfAssessment attribute
     * @return the formOfAssessment or null if not set
     */
    public RichText getFormOfAssessment()
    {
        return getValue(FORM_OF_ASSESSMENT);
    }
    
    /**
     * Get the syllabus
     * @return the syllabus or null if not set
     */
    public RichText getSyllabus()
    {
        return getValue(SYLLABUS);
    }
    
    /**
     * Get the list of LOM sheets
     * @return the list of LOMsheets or an empty list
     */
    public Set<LOMSheet> getLOMSheets()
    {
        Set<LOMSheet> lomSheets = new HashSet<>();
        
        ModelAwareRepeater attributeLOMSheets = getRepeater(LOM_SHEETS);
        if (attributeLOMSheets != null)
        {
            for (ModelAwareRepeaterEntry attributeLOMSheet : attributeLOMSheets.getEntries())
            {
                LOMSheet lomSheet = new LOMSheet(attributeLOMSheet.getValue(LOM_SHEET_URL, false, StringUtils.EMPTY), attributeLOMSheet.getValue(LOM_SHEET_LABEL, false, StringUtils.EMPTY));
                lomSheets.add(lomSheet);
            }
        }

        return lomSheets;
    }
    
    /**
     * Set the LOM sheets
     * @param sheets The LOM sheets
     */
    public void setLOMSheets (Set<LOMSheet> sheets)
    {
        // Remove old LOM sheets if exist
        removeValue(LOM_SHEETS);
        
        ModifiableModelAwareRepeater attributeLOMSheets = getRepeater(LOM_SHEETS, true);
        for (LOMSheet lomSheet : sheets)
        {
            ModifiableModelAwareRepeaterEntry attributeLOMSheet = attributeLOMSheets.addEntry();
            attributeLOMSheet.setValue(LOM_SHEET_URL, lomSheet.getUrl());
            attributeLOMSheet.setValue(LOM_SHEET_LABEL, lomSheet.getLabel());
        }
    }
    
    /**
     * Determines if the course has LOM sheet
     * @param lomSheet The LOM sheet to test
     * @return <code>true</code> if the course has LOM sheet
     */
    public boolean hasLOMSheet (LOMSheet lomSheet)
    {
        for (LOMSheet lom : getLOMSheets())
        {
            if (lom.getUrl().equals(lomSheet.getUrl()) && lom.getLabel().equals(lomSheet.getLabel()))
            {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Get the additional information
     * @return the additional information
     */
    public RichText getAdditionalInformations()
    {
        return getValue(ADDITIONAL_INFORMATIONS);
    }

    /**
     * Get the Erasmus code
     * @return the Erasmus code
     */
    public String getErasmusCode()
    {
        return ContentDataHelper.getContentIdFromContentData(this, ERASMUS_CODE);
    }

    /**
     * Get the teaching location
     * @return the teaching location
     */
    public String[] getTeachingLocation()
    {
        return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, TEACHING_LOCATION);
    }

    /**
     * Set the teaching location
     * @param teachingLocation the teaching location to set
     * @throws AmetysRepositoryException if failed to set attribute
     */
    public void setTeachingLocation(String[] teachingLocation) throws AmetysRepositoryException
    {
        setValue(Course.TEACHING_LOCATION, teachingLocation);
    }

    /**
     * Get the ECTS
     * @return the ECTS
     */
    public double getEcts()
    {
        return getValue(ECTS, false, 0D);
    }
    
    
    /**
     * Get the ECTS for a given educational path
     * @param ctxEducationalPath The educational path
     * @return the ECTS for this educational path
     */
    public double getEcts(EducationalPath ctxEducationalPath)
    {
        return getEcts(
            Optional.ofNullable(ctxEducationalPath)
                .map(List::of)
                .orElseGet(List::of)
        );
    }
    
    /**
     * Get the ECTS for given educational paths
     * @param ctxEducationalPaths The educational paths
     * @return the ECTS for this educational path
     */
    public double getEcts(List<EducationalPath> ctxEducationalPaths)
    {
        return Optional.ofNullable(ctxEducationalPaths)
            .filter(Predicate.not(List::isEmpty))
            .flatMap(paths -> _getFactory()._getODFHelper().<Double>getValueForPath(this, ECTS_BY_PATH, paths))
            .orElseGet(this::getEcts);
    }
    
    /**
     * Get the number of hours
     * @return the number of hours
     */
    public double getNumberOfHours()
    {
        return getValue(NUMBER_OF_HOURS, false, 0D);
    }

    /**
     * Get the effectives
     * @return the effectives
     * @throws AmetysRepositoryException if failed to get attribute
     */
    public String getMaxNumberOfStudents()
    {
        return getValue(MAX_NUMBER_OF_STUDENTS, false, StringUtils.EMPTY);
    }

    /**
     * Get the teaching term
     * @return the teaching term
     */
    public String getTeachingTerm()
    {
        return ContentDataHelper.getContentIdFromContentData(this, TEACHING_TERM);
    }

    /**
     * Get the level
     * @return the level
     */
    public String getLevel()
    {
        return ContentDataHelper.getContentIdFromContentData(this, LEVEL);
    }

    /**
     * Get the teaching method
     * @return the teaching method
     */
    public String getFormOfTeachingMethod()
    {
        return ContentDataHelper.getContentIdFromContentData(this, FORMODFTEACHING_METHOD);
    }

    /**
     * Get the teaching organizations
     * @return the teaching organizations
     */
    public String[] getFormOfTeachingOrgs()
    {
        return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, FORMOFTEACHING_ORG);
    }

    /**
     * Get the teaching activity
     * @return the teaching activity
     */
    public String getTeachingActivity()
    {
        return ContentDataHelper.getContentIdFromContentData(this, TEACHING_ACTIVITY);
    }

    /**
     * Get the teaching language
     * @return the teaching language
     */
    public String[] getTeachingLanguage()
    {
        return ContentDataHelper.getContentIdsArrayFromMultipleContentData(this, TEACHING_LANGUAGE);
    }
    
    /**
     * Get the start date
     * @return the start date
     */
    public LocalDate getStartDate()
    {
        return getValue(START_DATE);
    }

    /**
     * Get the time slot
     * @return the time slot
     */
    public String getTimeSlot()
    {
        return ContentDataHelper.getContentIdFromContentData(this, TIME_SLOT);
    }

    /**
     * Get the keywords
     * @return the keywords
     */
    public String[] getKeywords()
    {
        return getValue(KEYWORDS, false, ArrayUtils.EMPTY_STRING_ARRAY);
    }

    /**
     * Get the web link label
     * @return the web link label
     */
    public String getWebLinkLabel()
    {
        return getValue(WEB_LINK_LABEL, false, StringUtils.EMPTY);
    }

    /**
     * Get the web link URL
     * @return the web link URL
     */
    public String getWebLinkUrl()
    {
        return getValue(WEB_LINK_URL, false, StringUtils.EMPTY);
    }

    /**
     * Get the course type (nature)
     * @return the course type
     */
    public String getCourseType()
    {
        return ContentDataHelper.getContentIdFromContentData(this, COURSE_TYPE);
    }

    /**
     * Get the bibliography
     * @return the bibliography
     */
    public RichText getBibliography()
    {
        return getValue(BIBLIOGRAPHY);
    }
    
    /**
     * Get the skills
     * @return the skills
     */
    public RichText getSkills()
    {
        return getValue(SKILLS);
    }
    
    /**
     * Is open to exchange students
     * @return <code>true</code> if the course is open to exchange students
     */
    public boolean isOpenToExchangeStudents()
    {
        return getValue(OPEN_TO_EXCHANGE_STUDENTS, false, false);
    }
    
    // --------------------------------------------------------------------------------------//
    // CDM-fr
    // --------------------------------------------------------------------------------------//
    @Override
    public String getCDMId()
    {
        return "FRUAI" + _getFactory()._getRootOrgUnitRNE() + "CO" + getCode();
    }
    
    @Override
    public String getCdmCode()
    {
        return getValue(CDM_CODE, false, StringUtils.EMPTY);
    }
    
    @Override
    public void setCdmCode(String cdmCode)
    {
        setValue(CDM_CODE, cdmCode);
    }

    /**
     * Get the acquired skills for a given program
     * @param programId the Id of the program
     * @return The list of acquired skills for the program, or an empty array if there are no skills
     */
    public ContentValue[] getAcquiredSkills(String programId)
    {
        ModifiableIndexableRepeater repeater = getRepeater(ACQUIRED_MICRO_SKILLS);
        
        if (repeater != null)
        {
            ModifiableIndexableRepeaterEntry entryForProgram = repeater.getEntries().stream()
                                 .filter(entry -> entry.hasValue(ACQUIRED_MICRO_SKILLS_PROGRAM) && programId.equals(((ContentValue) entry.getValue(ACQUIRED_MICRO_SKILLS_PROGRAM, false, null)).getContentId()))
                                 .findFirst()
                                 .orElse(null);
            
            if (entryForProgram != null)
            {
                return entryForProgram.getValue(ACQUIRED_MICRO_SKILLS_SKILLS, true, new ContentValue[0]);
            }
        }
        
        return new ContentValue[0];
    }
}
