/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.helper;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.clientsideelement.content.SmartContentClientSideElementHelper;
import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.data.ContentDataHelper;
import org.ametys.cms.indexing.solr.SolrIndexHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableWorkflowAwareContent;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.trash.element.TrashElementDAO;
import org.ametys.cms.trash.model.TrashElementModel;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.cms.workflow.EditContentFunction;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.course.CourseFactory;
import org.ametys.odf.course.ShareableCourseConstants;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.observation.OdfObservationConstants;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.person.Person;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramPart;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.program.TraversableProgramPart;
import org.ametys.plugins.repository.AmetysObjectIterator;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableAmetysObject;
import org.ametys.plugins.repository.RemovableAmetysObject;
import org.ametys.plugins.repository.lock.LockableAmetysObject;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.repository.trash.TrashElement;
import org.ametys.plugins.repository.trash.TrashableAmetysObject;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.WorkflowException;

/**
 * Helper to delete an ODF content.
 */
public class DeleteODFContentHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** Avalon role. */
    public static final String ROLE = DeleteODFContentHelper.class.getName();
    
    /** Ametys object resolver */
    private AmetysObjectResolver _resolver;
    
    /** The ODF helper */
    private ODFHelper _odfHelper;
    
    /** Observer manager. */
    private ObservationManager _observationManager;
    
    /** The Content workflow helper */
    private ContentWorkflowHelper _contentWorkflowHelper;
    
    /** The current user provider */
    private CurrentUserProvider _currentUserProvider;
    
    /** The rights manager */
    private RightManager _rightManager;

    /** The content helper */
    private ContentHelper _contentHelper;
    
    /** Helper for smart content client elements */
    private SmartContentClientSideElementHelper _smartHelper;

    private SolrIndexHelper _solrIndexHelper;

    private TrashElementDAO _trashElementDAO;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _smartHelper = (SmartContentClientSideElementHelper) manager.lookup(SmartContentClientSideElementHelper.ROLE);
        _contentHelper = (ContentHelper) manager.lookup(ContentHelper.ROLE);
        _solrIndexHelper = (SolrIndexHelper) manager.lookup(SolrIndexHelper.ROLE);
        _trashElementDAO = (TrashElementDAO) manager.lookup(org.ametys.plugins.repository.trash.TrashElementDAO.ROLE);
    }
    
    /**
     * Enumeration for the mode of deletion
     *
     */
    public enum DeleteMode
    {
        /** Delete the content only */
        SINGLE,
        /** Delete the content and its structure */
        STRUCTURE_ONLY,
        /** Delete the content and its structure and its courses */
        FULL
    }

    /**
     * Trash contents if possible, or delete it.
     * @param contentsId The ids of contents to delete
     * @param modeParam The mode of deletion
     * @return the deleted and undeleted contents
     */
    public Map<String, Object> trashContents(List<String> contentsId, String modeParam)
    {
        return trashContents(contentsId, modeParam, false);
    }

    /**
     * Trash contents if possible, or delete it.
     * @param contentsId The ids of contents to delete
     * @param modeParam The mode of deletion
     * @param ignoreRights If true, bypass the rights check during the deletion
     * @return the deleted and undeleted contents
     */
    public Map<String, Object> trashContents(List<String> contentsId, String modeParam, boolean ignoreRights)
    {
        return _deleteContents(contentsId, modeParam, ignoreRights, false);
    }
    
    /**
     * Delete ODF contents
     * @param contentsId The ids of contents to delete
     * @param modeParam The mode of deletion
     * @return the deleted and undeleted contents
     */
    public Map<String, Object> deleteContents(List<String> contentsId, String modeParam)
    {
        return deleteContents(contentsId, modeParam, false);
    }

    /**
     * Delete ODF contents
     * @param contentsId The ids of contents to delete
     * @param modeParam The mode of deletion
     * @param ignoreRights If true, bypass the rights check during the deletion
     * @return the deleted and undeleted contents
     */
    public Map<String, Object> deleteContents(List<String> contentsId, String modeParam, boolean ignoreRights)
    {
        return _deleteContents(contentsId, modeParam, ignoreRights, true);
    }
    
    /**
     * Delete ODF contents
     * @param contentsId The ids of contents to delete
     * @param modeParam The mode of deletion
     * @param ignoreRights If true, bypass the rights check during the deletion
     * @param onlyDeletion <code>true</code> to really delete the contents, otherwise the contents will be trashed if trashable
     * @return the deleted and undeleted contents
     */
    private Map<String, Object> _deleteContents(List<String> contentsId, String modeParam, boolean ignoreRights, boolean onlyDeletion)
    {
        Map<String, Object> results = new HashMap<>();
        
        List<String> alreadyDeletedContentIds = new ArrayList<>();
        for (String contentId : contentsId)
        {
            Map<String, Object> result = new HashMap<>();
            result.put("deleted-contents", new HashSet<>());
            result.put("undeleted-contents", new HashSet<>());
            result.put("referenced-contents", new HashSet<>());
            result.put("unauthorized-contents", new HashSet<>());
            result.put("locked-contents", new HashSet<>());
            result.put("hierarchy-changed-contents", new HashMap<>());

            if (!alreadyDeletedContentIds.contains(contentId))
            {
                Content content = _resolver.resolveById(contentId);
                
                result.put("initial-content", content.getId());
                
                DeleteMode deleteMode = StringUtils.isNotBlank(modeParam) ? DeleteMode.valueOf(modeParam.toUpperCase()) : DeleteMode.SINGLE;
                
                boolean referenced = isContentReferenced(content);
                if (referenced || !_checkBeforeDeletion(content, deleteMode, ignoreRights, result))
                {
                    if (referenced)
                    {
                        // Indicate that the content is referenced.
                        @SuppressWarnings("unchecked")
                        Set<Content> referencedContents = (Set<Content>) result.get("referenced-contents");
                        referencedContents.add(content);
                    }
                    result.put("check-before-deletion-failed", true);
                }
                else
                {
                    // Process deletion
                    _deleteContent(content, deleteMode, ignoreRights, result, onlyDeletion);
                    
                    @SuppressWarnings("unchecked")
                    Set<String> deletedContents = (Set<String>) result.get("deleted-contents");
                    if (deletedContents != null)
                    {
                        alreadyDeletedContentIds.addAll(deletedContents);
                    }
                }
            }
            else
            {
                TrashElement trashElement = _trashElementDAO.find(contentId);
                if (trashElement != null && trashElement.<Boolean>getValue(TrashElementModel.HIDDEN))
                {
                    trashElement.setHidden(false);
                    trashElement.saveChanges();
                    
                    // Notify observers
                    Map<String, Object> eventParams = new HashMap<>();
                    eventParams.put(ObservationConstants.ARGS_TRASH_ELEMENT_ID, trashElement.getId());
                    eventParams.put(ObservationConstants.ARGS_AMETYS_OBJECT_ID, contentId);
                    _observationManager.notify(new Event(ObservationConstants.EVENT_TRASH_UPDATED, _currentUserProvider.getUser(), eventParams));
                }
            }
            
            results.put(contentId, result);
        }

        return results;
    }
    
    /**
     * Delete one content
     * @param content the content to delete
     * @param deleteMode The deletion mode
     * @param ignoreRights If true, bypass the rights check during the deletion
     * @param results the results map
     */
    private void _deleteContent(Content content, DeleteMode deleteMode, boolean ignoreRights, Map<String, Object> results, boolean onlyDeletion)
    {
        boolean success = true;
        
        if (content instanceof OrgUnit)
        {
            // 1- First delete relation to parent
            OrgUnit parentOrgUnit = ((OrgUnit) content).getParentOrgUnit();
            if (parentOrgUnit != null)
            {
                success = _removeRelation(parentOrgUnit, content, OrgUnit.CHILD_ORGUNITS, 22, results);
            }
            
            // 2 - If succeed, process to deletion
            if (success)
            {
                _deleteOrgUnit((OrgUnit) content, ignoreRights, results, onlyDeletion);
            }
        }
        else if (content instanceof ProgramItem)
        {
            // 1 - First delete relation to parents
            if (content instanceof Course)
            {
                List<CourseList> courseLists = ((Course) content).getParentCourseLists();
                success = _removeRelations(courseLists, content, CourseList.CHILD_COURSES, 22, results);
            }
            else if (content instanceof ProgramPart)
            {
                List<? extends ModifiableWorkflowAwareContent> parentProgramParts = ((ProgramPart) content).getProgramPartParents()
                                                                                                           .stream()
                                                                                                           .filter(ModifiableWorkflowAwareContent.class::isInstance)
                                                                                                           .map(ModifiableWorkflowAwareContent.class::cast)
                                                                                                           .collect(Collectors.toList());
                
                success = _removeRelations(parentProgramParts, content, TraversableProgramPart.CHILD_PROGRAM_PARTS, 22, results);
    
                if (success && content instanceof CourseList)
                {
                    List<Course> parentCourses = ((CourseList) content).getParentCourses();
                    success = _removeRelations(parentCourses, content, Course.CHILD_COURSE_LISTS, 22, results);
                }
            }
            else
            {
                throw new IllegalArgumentException("The content [" + content.getId() + "] is not of the expected type, it can't be deleted.");
            }
            
            // 2 - If succeed, process to deletion
            if (success)
            {
                _deleteProgramItem((ProgramItem) content, deleteMode, ignoreRights, results, onlyDeletion);
                
                // Notify observers for program items that parent relation has been removed
                @SuppressWarnings("unchecked")
                Map<String, List<? extends ProgramItem>> hierarchyChangedContents = (Map<String, List<? extends ProgramItem>>) results.get("hierarchy-changed-contents");
                for (Entry<String, List< ? extends ProgramItem>> entry : hierarchyChangedContents.entrySet())
                {
                    for (ProgramItem programItem : entry.getValue())
                    {
                        Map<String, Object> eventParams = new HashMap<>();
                        eventParams.put(OdfObservationConstants.ARGS_PROGRAM_ITEM, programItem);
                        eventParams.put(OdfObservationConstants.ARGS_PROGRAM_ITEM_ID, programItem.getId());
                        eventParams.put(OdfObservationConstants.ARGS_OLD_PARENT_PROGRAM_ITEM_ID, entry.getKey());
                        
                        _observationManager.notify(new Event(OdfObservationConstants.EVENT_PROGRAM_ITEM_HIERARCHY_CHANGED, _currentUserProvider.getUser(), eventParams));
                    }
                }
                
            }
        }
        else if (content instanceof Person)
        {
            // 1 - Process to deletion
            _finalizeDeleteContents(Collections.singleton(new DeletionInfo(content.getId(), List.of(), false)), content.getParent(), results, onlyDeletion);
        }
        else
        {
            throw new IllegalArgumentException("The content [" + content.getId() + "] is not of the expected type, it can't be deleted.");
        }

        if (!success)
        {
            @SuppressWarnings("unchecked")
            Set<Content> undeletedContents = (Set<Content>) results.get("undeleted-contents");
            undeletedContents.add(content);
        }
    }

    /**
     * Test if content is still referenced before removing it
     * @param content The content to remove
     * @return true if content is still referenced
     */
    public boolean isContentReferenced(Content content)
    {
        return _isContentReferenced(content, null);
    }
    
    /**
     * Test if content is still referenced before removing it.
     * @param content The content to remove
     * @param rootContent the initial content to delete (can be null if checkRoot is false)
     * @return true if content is still referenced
     */
    private boolean _isContentReferenced(Content content, Content rootContent)
    {
        if (content instanceof OrgUnit)
        {
            return _isReferencedOrgUnit((OrgUnit) content);
        }
        else if (content instanceof ProgramItem)
        {
            if (rootContent != null)
            {
                return _isReferencedContentCheckingRoot((ProgramItem) content, rootContent);
            }
            else
            {
                List<ProgramItem> ignoredRefContent = _odfHelper.getChildProgramItems((ProgramItem) content);
                if (!(content instanceof Program))
                {
                    ignoredRefContent.addAll(_odfHelper.getParentProgramItems((ProgramItem) content));
                }
                
                for (Pair<String, Content> refPair : _contentHelper.getReferencingContents(content))
                {
                    Content refContent = refPair.getValue();
                    String path = refPair.getKey();
                    
                    // Ignoring reference from shareable field
                    if (!(refContent instanceof Course && path.equals(ShareableCourseConstants.PROGRAMS_FIELD_ATTRIBUTE_NAME)))
                    {
                        // The ref content is not ignored
                        if (refContent instanceof ProgramItem programItem && !ignoredRefContent.contains(programItem))
                        {
                            return true;
                        }
                    }
                }

                return false;
            }
        }
        else if (content instanceof CoursePart)
        {
            if (rootContent != null)
            {
                // we don't rely on the CoursePart#getCourses method to avoid issue with dirty data
                String query = QueryHelper.getXPathQuery(null, CourseFactory.COURSE_NODETYPE, new StringExpression(Course.CHILD_COURSE_PARTS, Operator.EQ, content.getId()));
                AmetysObjectIterator<ProgramItem> iterator = _resolver.<ProgramItem>query(query).iterator();
                while (iterator.hasNext())
                {
                    if (_isReferencedContentCheckingRoot(iterator.next(), rootContent))
                    {
                        return true;
                    }
                    
                }
                
                return false;
            }
            // There shouldn't be a case were we try to delete a coursePart without deleting it from a Course.
            // But in case of we support it
            else
            {
                // Verify if the content has no parent courses
                if (((CoursePart) content).getCourses().isEmpty())
                {
                    // Twice...
                    String query = QueryHelper.getXPathQuery(null, CourseFactory.COURSE_NODETYPE, new StringExpression(Course.CHILD_COURSE_PARTS, Operator.EQ, content.getId()));
                    return _resolver.query(query).iterator().hasNext();
                }
                return true;
            }
        }
        
        return content.hasReferencingContents();
    }
    
    /**
     * True if the orgUnit is referenced
     * @param orgUnit the orgUnit
     * @return true if the orgUnit is referenced
     */
    private boolean _isReferencedOrgUnit(OrgUnit orgUnit)
    {
        OrgUnit parentOrgUnit = orgUnit.getParentOrgUnit();

        List<String> relatedOrgUnit = orgUnit.getSubOrgUnits();
        if (parentOrgUnit != null)
        {
            relatedOrgUnit.add(parentOrgUnit.getId());
        }
        
        for (Pair<String, Content> refPair : _contentHelper.getReferencingContents(orgUnit))
        {
            Content refContent = refPair.getValue();
            String path = refPair.getKey();
            
            // Ignoring reference from shareable field
            if (!(refContent instanceof Course && path.equals(ShareableCourseConstants.ORGUNITS_FIELD_ATTRIBUTE_NAME)))
            {
                if (!relatedOrgUnit.contains(refContent.getId()))
                {
                    return true;
                }
            }
        }
        
        return false;
    }

    /**
     * Check that deletion can be performed without blocking errors
     * @param content The initial content to delete
     * @param mode The deletion mode
     * @param ignoreRights If true, bypass the rights check during the deletion
     * @param results The results
     * @return true if the deletion can be performed
     */
    private boolean _checkBeforeDeletion(Content content, DeleteMode mode, boolean ignoreRights, Map<String, Object> results)
    {
        // Check right and lock on content it self
        boolean allRight = _canDeleteContent(content, ignoreRights, results);
        
        // Check lock on parent contents
        allRight = _checkParentsBeforeDeletion(content, results) && allRight;
        
        // Check right and lock on children to be deleted or modified
        allRight = _checkChildrenBeforeDeletion(content, content, mode, ignoreRights, results) && allRight;
        
        return allRight;
    }
    
    private boolean _checkParentsBeforeDeletion(Content content,  Map<String, Object> results)
    {
        boolean allRight = true;
        
        // Check if parents are not locked
        List< ? extends WorkflowAwareContent> parents = _getParents(content);
        for (WorkflowAwareContent parent : parents)
        {
            if (_smartHelper.isLocked(parent))
            {
                @SuppressWarnings("unchecked")
                Set<Content> lockedContents = (Set<Content>) results.get("locked-contents");
                lockedContents.add(content);
                
                allRight = false;
            }
        }
        
        return allRight;
    }
    
    /**
     * Browse children to check if deletion could succeed
     * @param rootContentToDelete The initial content to delete
     * @param contentToCheck The current content to check
     * @param mode The deletion mode
     * @param ignoreRights If true, bypass the rights check during the deletion
     * @param results The result
     * @return true if the deletion can be processed
     */
    private boolean _checkChildrenBeforeDeletion(Content rootContentToDelete, Content contentToCheck, DeleteMode mode, boolean ignoreRights, Map<String, Object> results)
    {
        boolean allRight = true;
        
        if (contentToCheck instanceof ProgramItem)
        {
            allRight = _checkChildrenBeforeDeletionOfProgramItem(rootContentToDelete, contentToCheck, mode, ignoreRights, results);
        }
        else if (contentToCheck instanceof OrgUnit)
        {
            allRight = _checkChildrenBeforeDeletionOfOrgUnit(rootContentToDelete, contentToCheck, mode, ignoreRights, results);
        }
        
        return allRight;
    }
    
    private boolean _checkChildrenBeforeDeletionOfProgramItem(Content rootContentToDelete, Content contentToCheck, DeleteMode mode,  boolean ignoreRights, Map<String, Object> results)
    {
        boolean allRight = true;

        List<ProgramItem> childProgramItems = _odfHelper.getChildProgramItems((ProgramItem) contentToCheck);
        for (ProgramItem childProgramItem : childProgramItems)
        {
            Content childContent = (Content) childProgramItem;
            if (_smartHelper.isLocked(childContent))
            {
                // Lock should be checked for all children
                @SuppressWarnings("unchecked")
                Set<Content> lockedContents = (Set<Content>) results.get("locked-contents");
                lockedContents.add(childContent);
                
                allRight = false;
            }

            // If content is not referenced it should be deleted, so check right
            if ((mode == DeleteMode.FULL
                    || mode == DeleteMode.STRUCTURE_ONLY && !(childProgramItem instanceof Course))
                 && !_isContentReferenced(childContent, rootContentToDelete)
                 && !ignoreRights
                 && !hasRight(childContent))
            {
                // User has no sufficient right
                @SuppressWarnings("unchecked")
                Set<Content> norightContents = (Set<Content>) results.get("unauthorized-contents");
                norightContents.add(childContent);
                
                allRight = false;
            }
            
            if (mode != DeleteMode.SINGLE && !(mode == DeleteMode.STRUCTURE_ONLY && childProgramItem instanceof Course))
            {
                // Browse children recursively
                allRight = _checkChildrenBeforeDeletion(rootContentToDelete, childContent, mode, ignoreRights, results) && allRight;
            }
        }
        
        return allRight;
    }
    
    private boolean _checkChildrenBeforeDeletionOfOrgUnit(Content rootContentToDelete, Content contentToCheck, DeleteMode mode, boolean ignoreRights, Map<String, Object> results)
    {
        boolean allRight = true;

        List<String> childOrgUnits = ((OrgUnit) contentToCheck).getSubOrgUnits();
        for (String childOrgUnitId : childOrgUnits)
        {
            OrgUnit childOrgUnit = _resolver.resolveById(childOrgUnitId);
            if (!_canDeleteContent(childOrgUnit, ignoreRights, results))
            {
                allRight = false;
            }
            else if (_isReferencedOrgUnit(childOrgUnit))
            {
                @SuppressWarnings("unchecked")
                Set<Content> referencedContents = (Set<Content>) results.get("referenced-contents");
                referencedContents.add(childOrgUnit);
                
                allRight = false;
            }
            else
            {
                // Browse children recursively
                allRight = _checkChildrenBeforeDeletion(rootContentToDelete, childOrgUnit, mode, ignoreRights, results) && allRight;
            }
        }
        
        return allRight;
    }

    /**
     * Remove the relations between the content and its contents list
     * @param contentsToEdit the contents to edit
     * @param refContentToRemove The referenced content to be removed from contents
     * @param attributeName The name of attribute holding the relationship
     * @param actionId The id of workflow action to edit the relation
     * @param results the results map
     * @return true if remove relation successfully
     */
    private boolean _removeRelations(List<? extends ModifiableWorkflowAwareContent> contentsToEdit, Content refContentToRemove, String attributeName, int actionId, Map<String, Object> results)
    {
        boolean success = true;
        
        for (ModifiableWorkflowAwareContent contentToEdit : contentsToEdit)
        {
            success = _removeRelation(contentToEdit, refContentToRemove, attributeName, actionId, results) && success;
        }
        
        return success;
    }
    
    /**
     * Remove the relation parent-child relation on content.
     * @param contentToEdit The content to modified
     * @param refContentToRemove The referenced content to be removed from content
     * @param attributeName The name of attribute holding the child or parent relationship
     * @param actionId The id of workflow action to edit the relation
     * @param results the results map
     * @return boolean true if remove relation successfully
     */
    private boolean _removeRelation(ModifiableWorkflowAwareContent contentToEdit, Content refContentToRemove, String attributeName, int actionId, Map<String, Object> results)
    {
        try
        {
            List<String> values = ContentDataHelper.getContentIdsListFromMultipleContentData(contentToEdit, attributeName);
            
            if (values.contains(refContentToRemove.getId()))
            {
                values.remove(refContentToRemove.getId());
                
                Map<String, Object> inputs = new HashMap<>();
                Map<String, Object> parameters = new HashMap<>();
                
                parameters.put(EditContentFunction.VALUES_KEY, Map.of(attributeName, values));
                // Unlock the content
                parameters.put(EditContentFunction.QUIT, true);
                inputs.put(AbstractWorkflowComponent.CONTEXT_PARAMETERS_KEY, parameters);
                // Do not edit the invert relation, we want to keep the relation on the content we are deleting
                inputs.put(EditContentFunction.INVERT_RELATION_ENABLED, false);
                
                _contentWorkflowHelper.doAction(contentToEdit, actionId, inputs);
            }
            
            return true;
        }
        catch (WorkflowException | AmetysRepositoryException e)
        {
            getLogger().error("Unable to remove relationship to content {} ({}) on content {} ({}) for metadata {}", refContentToRemove.getTitle(), refContentToRemove.getId(), contentToEdit.getTitle(), contentToEdit.getId(), attributeName, e);
            return false;
        }
    }
    
    /**
     * Delete a program item
     * @param item The program item to delete
     * @param mode The deletion mode
     * @param ignoreRights If true, bypass the rights check during the deletion
     * @param results The results
     */
    private void _deleteProgramItem(ProgramItem item, DeleteMode mode, boolean ignoreRights, Map<String, Object> results, boolean onlyDeletion)
    {
        if (mode == DeleteMode.SINGLE)
        {
            if (_canDeleteContent((Content) item, ignoreRights, results))
            {
                Set<DeletionInfo> toDelete = new HashSet<>();
                String idToDelete = item.getId();
                List<String> linkedContents = new ArrayList<>();
                
                toDelete.add(new DeletionInfo(idToDelete, linkedContents, false));
                // 1 - First remove relations with children
                String parentMetadataName;
                if (item instanceof Course course)
                {
                    parentMetadataName = CourseList.PARENT_COURSES;
                    linkedContents.addAll(course.getCourseParts().stream().map(Content::getId).toList());
                    toDelete.addAll(_getCoursePartsToDelete(course, item, results));
                }
                else if (item instanceof CourseList)
                {
                    parentMetadataName = Course.PARENT_COURSE_LISTS;
                }
                else
                {
                    parentMetadataName = ProgramPart.PARENT_PROGRAM_PARTS;
                }
                
                List<ProgramItem> childProgramItems = _odfHelper.getChildProgramItems(item);
                List<ModifiableWorkflowAwareContent> childContents = childProgramItems.stream()
                        .filter(ModifiableWorkflowAwareContent.class::isInstance)
                        .map(ModifiableWorkflowAwareContent.class::cast)
                        .collect(Collectors.toList());
                
                linkedContents.addAll(childContents.stream().map(Content::getId).toList());
                
                boolean success = _removeRelations(childContents, (Content) item, parentMetadataName, 22, results);
                if (success)
                {
                    // If success delete course
                    _finalizeDeleteContents(toDelete, item.getParent(), results, onlyDeletion);
                    
                    @SuppressWarnings("unchecked")
                    Set<String> deletedContents = (Set<String>) results.get("deleted-contents");
                    if (deletedContents.contains(idToDelete))
                    {
                        @SuppressWarnings("unchecked")
                        Map<String, List<? extends ProgramItem>> hierarchyChangedContents = (Map<String, List<? extends ProgramItem>>) results.get("hierarchy-changed-contents");
                        hierarchyChangedContents.put(idToDelete, childProgramItems);
                    }
                }
            }
        }
        else
        {
            Set<DeletionInfo> toDelete = _getChildrenIdToDelete(item, item, results, mode, ignoreRights);
            _finalizeDeleteContents(toDelete, item.getParent(), results, onlyDeletion);
        }
    }
    
    private Set<DeletionInfo> _getCoursePartsToDelete(Course course, ProgramItem initialContentToDelete, Map<String, Object> results)
    {
        Set<DeletionInfo> toDelete = new HashSet<>();
        for (CoursePart childCoursePart : course.getCourseParts())
        {
            // check if the coursePart is referenced
            if (!_isContentReferenced(childCoursePart, (Content) initialContentToDelete))
            {
                // we don't check if we can delete the coursePart as we have already check it's course
                // we can add it to the list of content that will be deleted
                toDelete.add(new DeletionInfo(childCoursePart.getId(), List.of(), true));
            }
            // the content is still referenced, so we remove the relation from the course part
            else
            {
                _removeRelation(childCoursePart, course, CoursePart.PARENT_COURSES, 22, results);
            }
        }
        return toDelete;
    }

    /**
     * Delete one orgUnit
     * @param orgUnit the orgUnit to delete
     * @param ignoreRights If true, bypass the rights check during the deletion
     * @param results the results map
     */
    @SuppressWarnings("unchecked")
    private void _deleteOrgUnit(OrgUnit orgUnit, boolean ignoreRights, Map<String, Object> results, boolean onlyDeletion)
    {
        Collection<DeletionInfo> deletionInfos = _getOrgUnitDeletionInfos(orgUnit, ignoreRights, true, results);
        
        Set<Content> referencedContents = (Set<Content>) results.get("referenced-contents");
        Set<Content> lockedContents = (Set<Content>) results.get("locked-contents");
        Set<Content> unauthorizedContents = (Set<Content>) results.get("unauthorized-contents");
        
        if (referencedContents.size() == 0 && lockedContents.size() == 0 && unauthorizedContents.size() == 0)
        {
            _finalizeDeleteContents(deletionInfos, orgUnit.getParent(), results, onlyDeletion);
        }
    }

    /**
     * Finalize the deletion of contents. Call observers and remove contents
     * @param toDelete the list of info for each deletion operation to perform
     * @param parent the jcr parent for saving changes
     * @param results the results map
     */
    private void _finalizeDeleteContents(Collection<DeletionInfo> toDelete, ModifiableAmetysObject parent, Map<String, Object> results, boolean onlyDeletion)
    {
        @SuppressWarnings("unchecked")
        Set<Content> unauthorizedContents = (Set<Content>) results.get("unauthorized-contents");
        @SuppressWarnings("unchecked")
        Set<Content> lockedContents = (Set<Content>) results.get("locked-contents");
        
        if (!unauthorizedContents.isEmpty() || !lockedContents.isEmpty())
        {
            //Do Nothing
            return;
        }
        
        try
        {
            _solrIndexHelper.pauseSolrCommitForEvents(new String[] {ObservationConstants.EVENT_CONTENT_DELETED});
            
            Map<String, Map<String, Object>> eventParams = new HashMap<>();
            for (DeletionInfo info : toDelete)
            {
                String id = info.contentId();
                Content content = _resolver.resolveById(id);
                Map<String, Object> eventParam = _getEventParametersForDeletion(content);
                
                eventParams.put(id, eventParam);
                
                _observationManager.notify(new Event(ObservationConstants.EVENT_CONTENT_DELETING, _currentUserProvider.getUser(), eventParam));
                
                // Remove the content.
                LockableAmetysObject lockedContent = (LockableAmetysObject) content;
                if (lockedContent.isLocked())
                {
                    lockedContent.unlock();
                }
                
                if (!onlyDeletion && content instanceof TrashableAmetysObject trashableAO)
                {
                    _trashElementDAO.trash(trashableAO, info.hidden(), info.linkedContents().toArray(String[]::new));
                }
                else
                {
                    ((RemovableAmetysObject) content).remove();
                }
            }
            
            parent.saveChanges();
            
            for (DeletionInfo info : toDelete)
            {
                _observationManager.notify(new Event(ObservationConstants.EVENT_CONTENT_DELETED, _currentUserProvider.getUser(), eventParams.get(info.contentId())));
                
                @SuppressWarnings("unchecked")
                Set<String> deletedContents = (Set<String>) results.get("deleted-contents");
                deletedContents.add(info.contentId());
            }
        }
        finally
        {
            _solrIndexHelper.restartSolrCommitForEvents(new String[] {ObservationConstants.EVENT_CONTENT_DELETED});
        }
    }
    
    /**
     * True if we can delete the content (check if removable, rights and if locked)
     * @param content the content
     * @param ignoreRights If true, bypass the rights check during the deletion
     * @param results the results map
     * @return true if we can delete the content
     */
    private boolean _canDeleteContent(Content content, boolean ignoreRights, Map<String, Object> results)
    {
        if (!(content instanceof RemovableAmetysObject))
        {
            throw new IllegalArgumentException("The content [" + content.getId() + "] is not a RemovableAmetysObject, it can't be deleted.");
        }
        
        if (!ignoreRights && !hasRight(content))
        {
            // User has no sufficient right
            @SuppressWarnings("unchecked")
            Set<Content> norightContents = (Set<Content>) results.get("unauthorized-contents");
            norightContents.add(content);
            
            return false;
        }
        else if (_smartHelper.isLocked(content))
        {
            @SuppressWarnings("unchecked")
            Set<Content> lockedContents = (Set<Content>) results.get("locked-contents");
            lockedContents.add(content);
            
            return false;
        }
        
        return true;
    }
    
    /**
     * Get parameters for content deleted {@link Event}
     * @param content the removed content
     * @return the event's parameters
     */
    private Map<String, Object> _getEventParametersForDeletion (Content content)
    {
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_CONTENT, content);
        eventParams.put(ObservationConstants.ARGS_CONTENT_NAME, content.getName());
        eventParams.put(ObservationConstants.ARGS_CONTENT_ID, content.getId());
        return eventParams;
    }
    
    private List<? extends WorkflowAwareContent> _getParents(Content content)
    {
        if (content instanceof OrgUnit)
        {
            return Collections.singletonList(((OrgUnit) content).getParentOrgUnit());
        }
        else if (content instanceof Course)
        {
            return ((Course) content).getParentCourseLists();
        }
        else if (content instanceof CourseList)
        {
            List<ProgramPart> parentProgramItems = ((CourseList) content).getProgramPartParents();
            List<WorkflowAwareContent> parents = parentProgramItems.stream().map(p -> (WorkflowAwareContent) p).collect(Collectors.toList());
            
            List<Course> parentCourses = ((CourseList) content).getParentCourses();
            parents.addAll(parentCourses.stream().map(p -> (WorkflowAwareContent) p).collect(Collectors.toList()));
            
            return parents;
        }
        else if (content instanceof ProgramPart)
        {
            List<ProgramItem> parentProgramItems = _odfHelper.getParentProgramItems((ProgramPart) content);
            return parentProgramItems.stream().map(p -> (WorkflowAwareContent) p).collect(Collectors.toList());
        }
        
        return Collections.EMPTY_LIST;
    }
    
    /**
     * Get the id of children to be deleted.
     * All children shared with other contents which are not part of deletion, will be not deleted.
     * @param orgUnit The orgunit to delete
     * @param ignoreRights If true, bypass the rights check during the deletion
     * @param results The results
     * @return The id of contents to be deleted
     */
    private Collection<DeletionInfo> _getOrgUnitDeletionInfos (OrgUnit orgUnit, boolean ignoreRights, boolean userTarget, Map<String, Object> results)
    {
        List<DeletionInfo> toDelete = new ArrayList<>();
        
        if (_canDeleteContent(orgUnit, ignoreRights, results))
        {
            List<String> subOrgUnits = orgUnit.getSubOrgUnits();
            toDelete.add(new DeletionInfo(orgUnit.getId(), subOrgUnits, !userTarget));
            
            for (String childId : subOrgUnits)
            {
                OrgUnit childOrgUnit = _resolver.resolveById(childId);
                
                if (!_isReferencedOrgUnit(orgUnit))
                {
                    toDelete.addAll(_getOrgUnitDeletionInfos(childOrgUnit, ignoreRights, false, results));
                }
                else
                {
                    // The child program item can not be deleted, list the relation to the parent and stop iteration
                    @SuppressWarnings("unchecked")
                    Set<Content> referencedContents = (Set<Content>) results.get("referenced-contents");
                    referencedContents.add(childOrgUnit);
                }
            }
        }
        
        return toDelete;
    }
    
    /**
     * Get the id of children to be deleted.
     * All children shared with other contents which are not part of deletion, will be not deleted.
     * @param contentToDelete The content to delete in the tree of initial content to delete
     * @param initialContentToDelete The initial content to delete
     * @param results The results
     * @param mode The deletion mode
     * @param ignoreRights If true, bypass the rights check during the deletion
     * @return The id of contents to be deleted
     */
    private Set<DeletionInfo> _getChildrenIdToDelete (ProgramItem contentToDelete, ProgramItem initialContentToDelete, Map<String, Object> results, DeleteMode mode, boolean ignoreRights)
    {
        Set<DeletionInfo> toDelete = new HashSet<>();
        
        if (_canDeleteContent((Content) contentToDelete, ignoreRights, results))
        {
            List<String> linkedContents = new ArrayList<>();
            toDelete.add(new DeletionInfo(contentToDelete.getId(), linkedContents, !contentToDelete.equals(initialContentToDelete)));
            
            // First we start by adding the coursePart if it's a Course
            if (contentToDelete instanceof Course course)
            {
                linkedContents.addAll(course.getCourseParts().stream().map(Content::getId).toList());
                toDelete.addAll(_getCoursePartsToDelete(course, initialContentToDelete, results));
            }
            
            List<ProgramItem> childProgramItems;
            if (mode == DeleteMode.STRUCTURE_ONLY && contentToDelete instanceof TraversableProgramPart)
            {
                // Get subprogram, container and course list children only
                childProgramItems = ((TraversableProgramPart) contentToDelete).getProgramPartChildren().stream().map(ProgramItem.class::cast).collect(Collectors.toList());
            }
            else
            {
                childProgramItems = _odfHelper.getChildProgramItems(contentToDelete);
            }
            
            for (ProgramItem childProgramItem : childProgramItems)
            {
                linkedContents.add(childProgramItem.getId());
                if (!_isContentReferenced((Content) childProgramItem, (Content) initialContentToDelete))
                {
                    // If all references of this program item is part of the initial content to delete, it can be deleted
                    if (mode == DeleteMode.STRUCTURE_ONLY && childProgramItem instanceof CourseList courseList)
                    {
                        // Remove the relations to the course list to be deleted on all child courses
                        List<Course> courses = courseList.getCourses();
                        toDelete.add(new DeletionInfo(childProgramItem.getId(), courses.stream().map(Content::getId).toList(), true));
                        _removeRelations(courses, (Content) childProgramItem, Course.PARENT_COURSE_LISTS, 22, results);
                        
                        @SuppressWarnings("unchecked")
                        Map<String, List<? extends ProgramItem>> hierarchyChangedContents = (Map<String, List<? extends ProgramItem>>) results.get("hierarchy-changed-contents");
                        hierarchyChangedContents.put(childProgramItem.getId(), ((CourseList) childProgramItem).getCourses());
                    }
                    else
                    {
                        // Browse children recursively
                        toDelete.addAll(_getChildrenIdToDelete(childProgramItem, initialContentToDelete, results, mode, ignoreRights));
                    }
                }
                else
                {
                    // The child program item can not be deleted, remove the relation to the parent and stop iteration
                    String parentMetadataName;
                    if (childProgramItem instanceof CourseList)
                    {
                        parentMetadataName = contentToDelete instanceof Course ? CourseList.PARENT_COURSES : ProgramPart.PARENT_PROGRAM_PARTS;
                    }
                    else if (childProgramItem instanceof Course)
                    {
                        parentMetadataName = Course.PARENT_COURSE_LISTS;
                    }
                    else
                    {
                        parentMetadataName = ProgramPart.PARENT_PROGRAM_PARTS;
                    }
                    
                    _removeRelation((ModifiableWorkflowAwareContent) childProgramItem, (Content) contentToDelete, parentMetadataName, 22, results);
                    
                    @SuppressWarnings("unchecked")
                    Set<Content> referencedContents = (Set<Content>) results.get("referenced-contents");
                    referencedContents.add((Content) childProgramItem);
                    
                    @SuppressWarnings("unchecked")
                    Map<String, List<? extends ProgramItem>> hierarchyChangedContents = (Map<String, List<? extends ProgramItem>>) results.get("hierarchy-changed-contents");
                    hierarchyChangedContents.put(contentToDelete.getId(), List.of(childProgramItem));
                }
            }
        }
        
        return toDelete;
    }

    /**
     * Determines if the user has sufficient right for the given content
     * @param content the content
     * @return true if user has sufficient right
     */
    public boolean hasRight(Content content)
    {
        String rightId = _getRightId(content);
        return _rightManager.hasRight(_currentUserProvider.getUser(), rightId, content) == RightResult.RIGHT_ALLOW;
    }
    
    private String _getRightId (Content content)
    {
        if (content instanceof Course)
        {
            return "ODF_Rights_Course_Delete";
        }
        else if (content instanceof SubProgram)
        {
            return "ODF_Rights_SubProgram_Delete";
        }
        else if (content instanceof Container)
        {
            return "ODF_Rights_Container_Delete";
        }
        else if (content instanceof Program)
        {
            return "ODF_Rights_Program_Delete";
        }
        else if (content instanceof Person)
        {
            return "ODF_Rights_Person_Delete";
        }
        else if (content instanceof OrgUnit)
        {
            return "ODF_Rights_OrgUnit_Delete";
        }
        else if (content instanceof CourseList)
        {
            return "ODF_Rights_CourseList_Delete";
        }
        return "CMS_Rights_DeleteContent";
    }
    
    /**
     * True if the content is referenced (we are ignoring parent references if they have same root)
     * @param programItem the program item
     * @param initialContentToDelete the initial content to delete
     * @return true if the content is referenced
     */
    private boolean _isReferencedContentCheckingRoot(ProgramItem programItem, Content initialContentToDelete)
    {
        if (programItem.getId().equals(initialContentToDelete.getId()))
        {
            return false;
        }

        List<ProgramItem> parentProgramItems = _odfHelper.getParentProgramItems(programItem);
        if (parentProgramItems.isEmpty())
        {
            // We have found the root parent of our item. but it's not the initial content to delete
            return true;
        }
        
        for (ProgramItem parentProgramItem : parentProgramItems)
        {
            if (_isReferencedContentCheckingRoot(parentProgramItem, initialContentToDelete))
            {
                return true;
            }
        }
        
        return false;
    }
    
    private record DeletionInfo(String contentId, Collection<String> linkedContents, boolean hidden) { }
}
