/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.observation;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.observation.AsyncObserver;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.odf.EducationalPathHelper;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.WorkflowException;


/**
 * {@link Observer} when a educational path is no more valid after moving or removing a program item in ODF tree
 */
public class EducationalPathRemovedObserver extends AbstractLogEnabled implements AsyncObserver, Serviceable
{
    /** The educational path helper */
    protected EducationalPathHelper _educationalPathHelper;
    /** The oDF helper */
    protected ODFHelper _odfHelper;
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;

    public void service(ServiceManager smanager) throws ServiceException
    {
        _educationalPathHelper = (EducationalPathHelper) smanager.lookup(EducationalPathHelper.ROLE);
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
    }

    public boolean supports(Event event)
    {
        return event.getId().equals(OdfObservationConstants.EVENT_PROGRAM_ITEM_HIERARCHY_CHANGED);
    }

    public int getPriority()
    {
        return MIN_PRIORITY;
    }

    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        ProgramItem programItem = getProgramItem(event);
        String oldParentProgramItemId = getOldParentProgramItemId(event);
        
        if (oldParentProgramItemId != null) // program item has been moved or removed
        {
            List<String> newParentProgramItems = getNewParentProgramItemIds(event);
            if (newParentProgramItems.isEmpty())
            {
                // The program item has been removed, so him or its children may become unshared
                _updateUnsharedProgramItem(programItem);
            }
            
            // Then, when a program item has been detached from its parent (but its is still shared), all educational paths refering it AND its old parent are invalids and should be removed
            _educationalPathHelper.removeEducationalPathReferences(List.of(programItem.getId(), oldParentProgramItemId));
        }
        
    }
    
    // FIXME For now a program item is considered as shared if it has several parents whatever its ancestors are orphans
    // So this method will do nothing except for the removed program item itself
    // Here it would actually not be necessary to browse child program items as their shared status will not change
    // When #isShared will ignore orphaned ancestors, this method will be the right way to do it
    private void _updateUnsharedProgramItem(ProgramItem programItem) throws WorkflowException
    {
        if (!_odfHelper.isShared(programItem))
        {
            // The program item is not shared anymore, all repeaters with educational path should be removed
            _educationalPathHelper.removeAllRepeatersWithEducationalPath(programItem);
        }
        
        List<ProgramItem> childProgramItems = _odfHelper.getChildProgramItems(programItem);
        for (ProgramItem childProgramItem : childProgramItems)
        {
            _updateUnsharedProgramItem(childProgramItem);
        }
    }
    
    /**
     * Get the {@link ProgramItem} concerned by this event
     * @param event the event
     * @return the program item content or null
     */
    protected ProgramItem getProgramItem(Event event)
    {
        Map<String, Object> args = event.getArguments();

        if (args.containsKey(OdfObservationConstants.ARGS_PROGRAM_ITEM_ID))
        {
            String programItemId = (String) args.get(OdfObservationConstants.ARGS_PROGRAM_ITEM_ID);
            return _resolver.resolveById(programItemId);
        }
        return null;
    }
    
    /**
     * Get the {@link ProgramItem} concerned by this event
     * @param event the event
     * @return the program item content or null
     */
    protected String getOldParentProgramItemId(Event event)
    {
        Map<String, Object> args = event.getArguments();

        if (args.containsKey(OdfObservationConstants.ARGS_OLD_PARENT_PROGRAM_ITEM_ID))
        {
            return (String) args.get(OdfObservationConstants.ARGS_OLD_PARENT_PROGRAM_ITEM_ID);
        }

        return null;
    }
    
    /**
     * Get the {@link ProgramItem} concerned by this event
     * @param event the event
     * @return the program item content or null
     */
    @SuppressWarnings("unchecked")
    protected List<String> getNewParentProgramItemIds(Event event)
    {
        Map<String, Object> args = event.getArguments();

        if (args.containsKey(OdfObservationConstants.ARGS_NEW_PARENT_PROGRAM_ITEM_IDS))
        {
            return (List<String>) args.get(OdfObservationConstants.ARGS_NEW_PARENT_PROGRAM_ITEM_IDS);
        }

        return List.of();
    }
}
