/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.observation.skill;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentDAO;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.odf.program.Program;
import org.ametys.odf.skill.ODFSkillsHelper;
import org.ametys.odf.skill.workflow.SkillEditionFunction;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Abstract observer to prepare skills observer when a content is updated
 */
public abstract class AbstractSkillsObserver extends AbstractLogEnabled implements Observer, Serviceable
{
    /** The content dao */
    protected ContentDAO _contentDAO;
    /** The content types helper */
    protected ContentTypesHelper _contentTypesHelper;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _contentDAO = (ContentDAO) manager.lookup(ContentDAO.ROLE);
        _contentTypesHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
    }
    
    @Override
    public int getPriority()
    {
        return MIN_PRIORITY - 3000;
    }
    
    public boolean supports(Event event)
    {
        if (ODFSkillsHelper.isSkillsEnabled() && event.getId().equals(getSupportedEventId()))
        {
            Content content = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
            return supportsContent(content);
        }
        
        return false;
    }
    
    /**
     * Get the supported event id
     * @return the event id
     */
    protected abstract String getSupportedEventId();
    
    /**
     * Check if the content is supported
     * @param content The content
     * @return a {@link Boolean}, <code>true</code> if the content is supported, <code>false</code> otherwise
     */
    protected boolean supportsContent(Content content)
    {
        // By default, we support Programs and Macro skills
        return content instanceof Program || _contentTypesHelper.isInstanceOf(content, SkillEditionFunction.MACRO_SKILL_TYPE);
    }
    
    /**
     * Retrieve the skill from a content
     * @param content The content
     * @return The skills
     */
    protected ContentValue[] _getSkills(Content content)
    {
        return content.getValue(getSkillsAttributeName(content));
    }

    /**
     * Get the skills attribute name
     * @param content The content
     * @return The attribute name for the skills
     */
    protected String getSkillsAttributeName(Content content)
    {
        // By default, we support Programs and Macro skills
        if (content instanceof Program)
        {
            return "skills";
        }
        else if (_contentTypesHelper.isInstanceOf(content, SkillEditionFunction.MACRO_SKILL_TYPE))
        {
            return "microSkills";
        }
        
        return "skills";
    }
}
