/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.properties.section.content;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Predicate;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.LocaleUtils;

import org.ametys.cms.properties.section.AbstractDefaultPropertySection;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.Content.ReferencingContentsSearch;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * The referencer items for {@link ProgramItem}.
 * It displays educational paths, children and other references to the content.
 */
public class ProgramItemReferencerSection extends AbstractDefaultPropertySection implements Serviceable
{
    private static final ListProgramItemComparator __LIST_PROGRAM_ITEM_COMPARATOR = new ListProgramItemComparator();

    private AmetysObjectResolver _resolver;
    private ODFHelper _odfHelper;

    public void service(ServiceManager smanager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
    }
    
    public boolean supports(AmetysObject ametysObject)
    {
        return ametysObject instanceof ProgramItem;
    }
    
    @Override
    protected Map<String, Object> buildData(AmetysObject ametysObject)
    {
        ProgramItem programItem = (ProgramItem) ametysObject;
        
        Map<String, Object> resultMap = new LinkedHashMap<>();

        resultMap.put("educationalPaths", _getEducationalPaths(programItem));
        resultMap.put("children", _getChildren(programItem));
        resultMap.put("otherReferences", _getOtherReferences((Content) ametysObject));
        
        return resultMap;
    }

    private List<List<Map<String, Object>>> _getEducationalPaths(ProgramItem programItem)
    {
        // To avoid to return path only with the program item itself
        if (_odfHelper.hasParentProgramItems(programItem))
        {
            return _odfHelper.getEducationalPaths(programItem)
                .stream()
                .map(p -> p.getProgramItems(_resolver))
                .sorted(__LIST_PROGRAM_ITEM_COMPARATOR)
                .map(this::_programItemList2JSON)
                .toList();
        }
        
        return List.of();
    }
    
    private List<Map<String, Object>> _getChildren(ProgramItem programItem)
    {
        List<Map<String, Object>> children = new ArrayList<>();
        
        // Add child program items to children
        children.addAll(_programItemList2JSON(_odfHelper.getChildProgramItems(programItem)));
        
        // Add course parts to children only if the content is a course
        if (programItem instanceof Course course)
        {
            children.addAll(_coursePartList2JSON(course.getCourseParts()));
        }
        
        return children;
    }

    private Map<String, Object> _getOtherReferences(Content content)
    {
        Map<String, Object> resultMap = new LinkedHashMap<>();
        
        ReferencingContentsSearch referencingContents = content.searchReferencingContents(100);

        // Build the list of already referenced contents
        List<Object> alreadyReferencedContents = new ArrayList<>();
        alreadyReferencedContents.addAll(_odfHelper.getChildProgramItems((ProgramItem) content));
        if (content instanceof Course course)
        {
            alreadyReferencedContents.addAll(course.getCourseParts());
        }
        alreadyReferencedContents.addAll(_odfHelper.getParentProgramItems((ProgramItem) content));
        
        Locale locale = Optional.of(content)
            .map(Content::getLanguage)
            .map(LocaleUtils::toLocale)
            .orElse(null);
        
        resultMap.put(
            "referencingContents",
            referencingContents.referencingContents()
                .stream()
                // Remove already referenced contents (direct children and direct parents)
                .filter(Predicate.not(alreadyReferencedContents::contains))
                .map(c -> _content2JSON(c, locale))
                .toList()
        );
        
        if (referencingContents.hasOtherReferences())
        {
            resultMap.put("hasOther", true);
        }
        return resultMap;
    }
    
    private List<Map<String, Object>> _programItemList2JSON(List<ProgramItem> programItems)
    {
        return _odfItemList2JSON(programItems, c -> ((ProgramItem) c).getDisplayCode());
    }
    
    private List<Map<String, Object>> _coursePartList2JSON(List<CoursePart> courseParts)
    {
        return _odfItemList2JSON(courseParts, c -> ((CoursePart) c).getDisplayCode());
    }
    
    private List<Map<String, Object>> _odfItemList2JSON(List<?> contents, Function<Content, String> getDisplayCodeFn)
    {
        return contents.stream()
                .map(Content.class::cast)
                .map(c -> _odfItem2JSON(c, getDisplayCodeFn))
                .toList();
    }
    
    private Map<String, Object> _odfItem2JSON(Content content, Function<Content, String> getDisplayCodeFn)
    {
        Map<String, Object> json = new HashMap<>();
        json.put("id", content.getId());
        json.put("title", content.getTitle());
        json.put("code", getDisplayCodeFn.apply(content));
        return json;
    }
    
    private Map<String, Object> _content2JSON(Content content, Locale locale)
    {
        return Map.of(
            "id", content.getId(),
            "title", content.getTitle(locale)
        );
    }

    /**
     * Comparator of {@link List} composing of {@link ProgramItem} by title.
     */
    private static final class ListProgramItemComparator implements Comparator<List<ProgramItem>>
    {
        public int compare(List<ProgramItem> l1, List<ProgramItem> l2)
        {
            int l1Size = l1.size();
            int l2Size = l2.size();
            
            for (int i = 0; i < l1Size; i++)
            {
                if (l2Size <= i)
                {
                    // l1 is greater than l2
                    return 1;
                }
                else
                {
                    // Compare title
                    int comparing = ((Content) l1.get(i)).getTitle().compareTo(((Content) l2.get(i)).getTitle());
                    if (comparing != 0)
                    {
                        return comparing;
                    }
                }
            }
            
            // Lists are the same at the end of reading l1
            // Then compare size of the lists
            return Integer.compare(l1Size, l2Size);
        }
    }
}
