/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.workflow.copy;

import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.content.CopyContentComponent;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.repository.ModifiableWorkflowAwareContent;
import org.ametys.cms.workflow.copy.CreateContentByCopyFunction;
import org.ametys.odf.ODFHelper;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.collection.AmetysObjectCollection;
import org.ametys.plugins.repository.data.holder.values.SynchronizableValue;

import com.opensymphony.workflow.WorkflowException;

/**
 * OSWorkflow function to create a odf content by copy of another
 */
public abstract class AbstractCreateODFContentByCopyFunction extends CreateContentByCopyFunction
{
    /** Key for the parent of the target content */
    public static final String PARENT_KEY = AbstractCreateProgramItemByCopyFunction.class.getName() + "$parentId";
    
    /** ODF helper */
    protected ODFHelper _odfHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        
        // Override component for copy
        _copyContentComponent = (CopyContentComponent) manager.lookup(org.ametys.odf.content.CopyContentComponent.ROLE);
    }
    
    @Override
    protected AmetysObjectCollection< ? , ModifiableWorkflowAwareContent> _getContentRoot(Map transientVars) throws WorkflowException
    {
        return _odfHelper.getRootContent(true);
    }
    
    @Override
    protected ModifiableWorkflowAwareContent _createContent(Map transientVars, Map args, String desiredContentName, ModifiableTraversableAmetysObject contentsNode)
    {
        String baseName = desiredContentName;
        String prefix = _getContentNamePrefix();
        if (!baseName.startsWith(prefix))
        {
            baseName = prefix + baseName;
        }

        return super._createContent(transientVars, args, baseName, contentsNode);
    }

    @Override
    protected void processValues(Map transientVars, ModifiableContent targetContent, Map<String, Object> values) throws WorkflowException
    {
        super.processValues(transientVars, targetContent, values);
        
        // Add the parent content if exists and compatible
        Content parent = _getParent(transientVars);
        
        if (parent != null && _isCompatibleParent(parent))
        {
            values.put(_getParentAttributeName(parent), _buildParentSynchronizableValue(parent));
        }
    }
    
    /**
     * Get the parent content if exists.
     * @param transientVars The transient vars
     * @return the parent content (may be not compatible at this stage)
     */
    protected Content _getParent(Map transientVars)
    {
        return Optional.of(PARENT_KEY)
            .map(transientVars::get)
            .map(String.class::cast)
            .map(_resolver::<Content>resolveById)
            .orElse(null);
    }
    
    /**
     * Check if the parent is compatible
     * @param parent The parent
     * @return <code>true</code> if the parent is compatible
     */
    protected abstract boolean _isCompatibleParent(Content parent);
    
    /**
     * Get the parent attribute name.
     * @param parent The parent
     * @return the parent attribute name
     */
    protected abstract String _getParentAttributeName(Content parent);
    
    /**
     * Get the parent attribute synchronizable value.
     * @param parent The parent
     * @return a synchronizable value
     */
    protected abstract SynchronizableValue _buildParentSynchronizableValue(Content parent);
    
    /**
     * Get the prefix for content name
     * @return the prefix
     */
    protected abstract String _getContentNamePrefix ();
}
