/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.odf;

import java.util.Collection;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.cocoon.acting.AbstractAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.core.util.StringUtils;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.config.Config;

/**
 * Abstract class for ODF authentication by IP (useful for CDM-fr portals).
 */
public abstract class AbstractODFAuthenticateAction extends AbstractAction
{
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        authenticate(request);
        return null;
    }
    
    /**
     * Method to authenticate
     * @param request The request
     */
    protected void authenticate(Request request)
    {
        // Verify the IP address
        String conf = Config.getInstance().getValue(getConfigParameter());
        Collection<String> ips = StringUtils.stringToCollection(conf);

        // The real client IP may have been put in the non-standard
        // "X-Forwarded-For" request header, in case of reverse proxy
        String xff = request.getHeader("X-Forwarded-For");
        String ip = null;

        if (xff != null)
        {
            ip = xff.split(",")[0];
        }
        else
        {
            ip = request.getRemoteAddr();
        }

        if (!ips.contains(ip))
        {
            throw new AccessDeniedException(String.format(getExceptionMessage(), ip));
        }
    }
    
    /**
     * Get the config parameter name contained the allowed IP adresses.
     * @return The parameter name
     */
    protected abstract String getConfigParameter();
    
    /**
     * Get the exception message if the current IP is not allowed.
     * @return The exception message
     */
    protected abstract String getExceptionMessage();
}

