/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.course.generators;

import java.io.IOException;
import java.util.Locale;
import java.util.Set;

import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.Generator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.odf.NoLiveVersionException;
import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.generators.ProgramContentGenerator;
import org.ametys.runtime.config.Config;

/**
 * {@link Generator} for rendering raw content data for a {@link Course}.
 */
public class CourseContentGenerator extends ProgramContentGenerator
{
    @Override
    protected void _saxOtherData(Content content, Locale defaultLocale) throws SAXException, ProcessingException, IOException
    {
        super._saxOtherData(content, defaultLocale);
        
        boolean isEdition = parameters.getParameterAsBoolean("isEdition", false);
        if (!isEdition)
        {
            if (content instanceof Course)
            {
                Course course = (Course) content;
                
                if (_odfHelper.useLegacyProgramStructure(course))
                {
                    // Child courses lists
                    saxCourseLists(course, defaultLocale);
                    
                    // Course parts
                    saxCourseParts(course, defaultLocale);
                }
                
                // Contacts
                saxPersons(course);
                
                // OrgUnits
                saxOrgUnits(course);
                
                // Translations
                saxTranslation(course);
                
                // Referenced programs 
                saxReferencedPrograms(course);
            }
        }
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        request.setAttribute(Content.class.getName(), content);
    }
    
    /**
     * SAX the referenced programs
     * @param course The course
     * @throws SAXException if an error occurs while saxing
     */
    protected void saxReferencedPrograms (Course course) throws SAXException
    {
        boolean showReferences = Config.getInstance().getValue("odf.course.showReferences");
        if (showReferences)
        {
            Set<Program> referencingPrograms = _odfHelper.getParentPrograms(course);
            for (Program program : referencingPrograms)
            {
                AttributesImpl attrs = new AttributesImpl();
                attrs.addCDATAAttribute("id", program.getId());
                XMLUtils.createElement(contentHandler, "refProgram", attrs, program.getTitle());
            }
        }
    }
    
    /**
     * SAX the referenced {@link CourseList}s
     * @param course The course
     * @param defaultLocale The default locale
     * @throws SAXException if an error occurs while saxing
     * @throws IOException if an error occurs
     * @throws ProcessingException if an error occurs
     */
    protected void saxCourseLists (Course course, Locale defaultLocale) throws SAXException, ProcessingException, IOException
    {
        for (CourseList cl: course.getCourseLists())
        {
            try
            {
                _odfHelper.switchToLiveVersionIfNeeded(cl);
                
                saxCourseList(cl, course.getContextPath(), defaultLocale);
            }
            catch (NoLiveVersionException e) 
            {
                // Just ignore the course list
            }
        }
    }

    /**
     * SAX the referenced {@link CoursePart}s
     * @param course The course
     * @param defaultLocale The default locale
     * @throws SAXException if an error occurs while saxing
     * @throws IOException if an error occurs
     * @throws ProcessingException if an error occurs
     */
    protected void saxCourseParts(Course course, Locale defaultLocale) throws SAXException, ProcessingException, IOException
    {
        for (CoursePart coursePart : course.getCourseParts())
        {
            try
            {
                _odfHelper.switchToLiveVersionIfNeeded(coursePart);
                
                saxCoursePart(coursePart, defaultLocale);
            }
            catch (NoLiveVersionException e) 
            {
                // Just ignore the course part
            }
        }
    }
}
