/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.program;

import java.util.Collections;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.odf.course.CourseFactory;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.plugins.repository.jcr.JCRTraversableAmetysObject;
import org.ametys.plugins.repository.jcr.TraversableAmetysObjectHelper;
import org.ametys.plugins.workflow.repository.WorkflowAwareAmetysObject;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.plugins.workflow.support.WorkflowProvider.AmetysObjectWorkflow;

import com.opensymphony.workflow.WorkflowException;
import com.opensymphony.workflow.spi.Step;

/**
 * {@link ProgramFactory} for handling {@link Program}
 */
public class ProgramFactory extends ProgramPartFactory
{
    /** {@link Program} nodetype for resources collection */
    public static final String PROGRAM_NODETYPE = RepositoryConstants.NAMESPACE_PREFIX + ":programContent";

    /** {@link Program} content type */
    public static final String PROGRAM_CONTENT_TYPE = "org.ametys.plugins.odf.Content.program";

    private WorkflowProvider _workflowProvider;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _workflowProvider = (WorkflowProvider) manager.lookup(WorkflowProvider.ROLE);
    }
    
    @Override
    public AbstractProgram getAmetysObject(Node node, String parentPath) throws AmetysRepositoryException
    {
        return new Program(node, parentPath, this);
    }
    
    /**
     * Returns the {@link AmetysObject} at the given subPath, relative to the
     * given {@link DefaultTraversableAmetysObject}.
     * 
     * @param <A> the actual type of {@link AmetysObject}.
     * @param object the context {@link DefaultTraversableAmetysObject}.
     * @param path the sub path. Cannot be <code>null</code>, empty or absolute.
     * @return the {@link AmetysObject} at the given subPath, relative to the
     *         given {@link DefaultTraversableAmetysObject}.
     * @throws AmetysRepositoryException if an error occurs.
     * @throws UnknownAmetysObjectException if no such object exists.
     */
    public <A extends AmetysObject> A getChild(JCRTraversableAmetysObject object, String path) throws AmetysRepositoryException, UnknownAmetysObjectException
    {
        return TraversableAmetysObjectHelper.<A> getChild(object, this, path, _resolver, getLogger());
    }

    /**
     * Returns all children of the given {@link DefaultTraversableAmetysObject}.
     * 
     * @param <A> the actual type of {@link AmetysObject}s
     * @param object a {@link DefaultTraversableAmetysObject}.
     * @return a List containing all children object in the Ametys hierarchy.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public <A extends AmetysObject> AmetysObjectIterable<A> getChildren(JCRTraversableAmetysObject object) throws AmetysRepositoryException
    {
        return TraversableAmetysObjectHelper.getChildren(object, this, _resolver, getLogger());
    }

    /**
     * Tests if a given object has a child with a given name.
     * 
     * @param object the context object.
     * @param name the name to test.
     * @return <code>true</code> is the given object has a child with the given
     *         name, <code>false</code> otherwise.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public boolean hasChild(JCRTraversableAmetysObject object, String name) throws AmetysRepositoryException
    {
        return TraversableAmetysObjectHelper.hasChild(object, name, _ametysFactoryExtensionPoint, getLogger());
    }

    /**
     * Creates a child to the given object.
     * 
     * @param <A> the actual type of {@link AmetysObject}.
     * @param object the parent {@link AmetysObject}.
     * @param name the new object's name.
     * @param type the new object's type.
     * @return the newly created {@link AmetysObject}.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public <A extends AmetysObject> A createChild(JCRTraversableAmetysObject object, String name, String type) throws AmetysRepositoryException
    {
        return TraversableAmetysObjectHelper.<A> createChild(object, this, name, type, _ametysFactoryExtensionPoint, _resolver, getLogger());
    }
    
    void recreateContentWorkflow(Node contentNode)
    {
        try
        {
            String workflowName = null;
            if (contentNode.isNodeType(ProgramFactory.PROGRAM_NODETYPE)
                || contentNode.isNodeType(SubProgramFactory.SUBPROGRAM_NODETYPE)
                || contentNode.isNodeType(ContainerFactory.CONTAINER_NODETYPE))
            {
                workflowName = "content";
            }
            else if (contentNode.isNodeType(CourseFactory.COURSE_NODETYPE))
            {
                workflowName = "course";
            }
            
            if (workflowName != null)
            {
                _createContentWorkflow(contentNode, workflowName);
            }
        }
        catch (WorkflowException | RepositoryException e)
        {
            getLogger().error("Impossible to recreate a workflow instance for content " + contentNode, e);
        }
    }
    
    private void _createContentWorkflow(Node contentNode, String workflowName) throws WorkflowException, RepositoryException
    {
        WorkflowAwareAmetysObject waAmetysObject = _resolver.resolve(contentNode, false);
        AmetysObjectWorkflow workflow = _workflowProvider.getAmetysObjectWorkflow(waAmetysObject);
        
        // initialize workflow
        long workflowId = workflow.initialize(workflowName, 0, Collections.emptyMap());
        
        // set workflow id + current step on the content
        waAmetysObject.setWorkflowId(workflowId);
        
        Step currentStep = (Step) workflow.getCurrentSteps(workflowId).iterator().next();
        waAmetysObject.setCurrentStepId(currentStep.getStepId());
    }
}
