/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.rights;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.DefaultContent;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.AccessController;
import org.ametys.core.right.AccessController.Permission.PermissionType;
import org.ametys.core.right.AccessExplanation;
import org.ametys.core.right.RightProfilesDAO;
import org.ametys.core.right.RightsException;
import org.ametys.core.user.UserIdentity;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.CourseFactory;
import org.ametys.odf.courselist.CourseListFactory;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.coursepart.CoursePartFactory;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.orgunit.OrgUnitFactory;
import org.ametys.odf.program.ContainerFactory;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.SubProgramFactory;
import org.ametys.odf.tree.ODFContentsTreeHelper;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.UserExpression;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;

/**
 * This access controller give access the content's creator, regardless of the required right,
 * if and only if the ODF content is still orphan (during creation process for example)
 */
public class ODFOrphanContentAccessController extends AbstractLogEnabled implements AccessController, Serviceable, PluginAware
{
    /** The right profile DAO */
    protected RightProfilesDAO _profileDAO;
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The ODF contents tree helper */
    protected ODFContentsTreeHelper _odfContentsTreeHelper;
    private ODFHelper _odfHelper;
    private String _id;

    public void service(ServiceManager manager) throws ServiceException
    {
        _odfContentsTreeHelper = (ODFContentsTreeHelper) manager.lookup(ODFContentsTreeHelper.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _profileDAO = (RightProfilesDAO) manager.lookup(RightProfilesDAO.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _id = id;
    }
    
    public String getId()
    {
        return _id;
    }
    
    public boolean supports(Object object)
    {
        return object instanceof ProgramItem || object instanceof OrgUnit || object instanceof CoursePart;
    }
    
    /**
     * Determines if the object is a orphan program item (without parent)
     * @param object the object
     * @return true if the object is a orphan program item
     */
    protected boolean _isOrphan(Object object)
    {
        if (object instanceof ProgramItem)
        {
            List<ProgramItem> parentProgramItems = _odfHelper.getParentProgramItems((ProgramItem) object);
            return parentProgramItems.isEmpty();
        }
        else if (object instanceof OrgUnit)
        {
            return ((OrgUnit) object).getParentOrgUnit() == null;
        }
        else if (object instanceof CoursePart)
        {
            return ((CoursePart) object).getCourses().isEmpty();
        }
        
        return false;
    }
    
    /**
     * Get the user permission on object
     * @param user the user
     * @param object the object
     * @return The access result
     */
    protected AccessResult _getUserPermission(UserIdentity user, Object object)
    {
        if (_isOrphan(object) && !_hasOrgUnit(object))
        {
            if (user.equals(((Content) object).getCreator()))
            {
                return AccessResult.USER_ALLOWED;
            }
        }
        return AccessResult.UNKNOWN;
    }
    
    /**
     * Determines if the object has a orgunit
     * @param object the object
     * @return true if the object is attach to a orgunit
     */
    protected boolean _hasOrgUnit(Object object)
    {
        if (object instanceof ProgramItem programItem)
        {
            return !programItem.getOrgUnits().isEmpty();
        }
        return false;
    }
    
    /**
     * Get the permission by users
     * @param object the object
     * @return the permission by users
     */
    protected Map<UserIdentity, AccessResult> _getPermissionByUser(Object object)
    {
        Map<UserIdentity, AccessResult> permissions = new HashMap<>();
        if (_isOrphan(object) && !_hasOrgUnit(object))
        {
            permissions.put(((Content) object).getCreator(), AccessResult.USER_ALLOWED);
        }
        return permissions;
    }

    @Override
    public AccessResult getPermission(UserIdentity user, Set<GroupIdentity> userGroups, String rightId, Object object)
    {
        return _getUserPermission(user, object);
    }

    @Override
    public AccessResult getReadAccessPermission(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        return _getUserPermission(user, object);
    }

    @Override
    public Map<String, AccessResult> getPermissionByRight(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        return Collections.EMPTY_MAP;
    }

    @Override
    public AccessResult getPermissionForAnonymous(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnonymous(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    @Override
    public AccessResult getPermissionForAnyConnectedUser(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    @Override
    public AccessResult getReadAccessPermissionForAnyConnectedUser(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    @Override
    public Map<UserIdentity, AccessResult> getPermissionByUser(String rightId, Object object)
    {
        return _getPermissionByUser(object);
    }

    @Override
    public Map<UserIdentity, AccessResult> getReadAccessPermissionByUser(Object object)
    {
        return _getPermissionByUser(object);
    }

    @Override
    public Map<GroupIdentity, AccessResult> getPermissionByGroup(String rightId, Object object)
    {
        return Collections.EMPTY_MAP;
    }

    @Override
    public Map<GroupIdentity, AccessResult> getReadAccessPermissionByGroup(Object object)
    {
        return Collections.EMPTY_MAP;
    }

    @Override
    public boolean hasUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups, String rightId)
    {
        return false;
    }

    @Override
    public boolean hasUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups)
    {
        return false;
    }

    @Override
    public boolean hasAnonymousAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    @Override
    public boolean hasAnonymousAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }

    @Override
    public boolean hasAnyConnectedUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    @Override
    public boolean hasAnyConnectedUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }
    
    public AccessExplanation explainReadAccessPermission(UserIdentity user, Set<GroupIdentity> groups, Object object)
    {
        return _getAccessExplanation(getReadAccessPermission(user, groups, object), object);
    }
    
    public AccessExplanation explainPermission(UserIdentity user, Set<GroupIdentity> groups, String rightId, Object object)
    {
        return _getAccessExplanation(getPermission(user, groups, rightId, object), object);
    }
    
    private AccessExplanation _getAccessExplanation(AccessResult permission, Object object)
    {
        if (permission == AccessResult.USER_ALLOWED)
        {
            return new AccessExplanation(getId(), permission,
                    new I18nizableText("plugin.odf", "PLUGINS_ODF_ORPHAN_ACCESS_CONTROLLER_EXPLANATION", _getI18nParameters(object))
                    );
        }
        return AccessController.getDefaultAccessExplanation(getId(), permission);
    }
    
    /**
     * Get the context label
     * @param object the context
     * @return the label
     */
    protected Map<String, I18nizableTextParameter> _getI18nParameters(Object object)
    {
        Map<String, I18nizableTextParameter> params = new HashMap<>();
        if (object instanceof Content content)
        {
            params.put("title", new I18nizableText(content.getTitle()));
        }
        
        if (object instanceof ProgramItem item)
        {
            params.put("code", new I18nizableText(item.getDisplayCode()));
        }
        else if (object instanceof OrgUnit orgunit)
        {
            params.put("code", new I18nizableText(orgunit.getDisplayCode()));
        }
        else if (object instanceof CoursePart part)
        {
            params.put("code", new I18nizableText(part.getDisplayCode()));
        }
        return params;
    }
    
    @Override
    public Map<ExplanationObject, Map<Permission, AccessExplanation>> explainAllPermissions(UserIdentity identity, Set<GroupIdentity> groups, Set<Object> workspacesContexts)
    {
        Map<ExplanationObject, Map<Permission, AccessExplanation>> result = new HashMap<>();
        
        if (workspacesContexts.contains("/cms"))
        {
            
            UserExpression userExpression = new UserExpression(DefaultContent.METADATA_CREATOR, Operator.EQ, identity);
            Expression contentTypeExpression = new ContentTypeExpression(Operator.EQ, ProgramFactory.PROGRAM_CONTENT_TYPE, SubProgramFactory.SUBPROGRAM_CONTENT_TYPE, CourseFactory.COURSE_CONTENT_TYPE, ContainerFactory.CONTAINER_CONTENT_TYPE, CoursePartFactory.COURSE_PART_CONTENT_TYPE, CourseListFactory.COURSE_LIST_CONTENT_TYPE, OrgUnitFactory.ORGUNIT_CONTENT_TYPE);
            String query = ContentQueryHelper.getContentXPathQuery(new AndExpression(userExpression, contentTypeExpression));
            AmetysObjectIterable<Content> contents = _resolver.query(query);
            
            for (Content content : contents)
            {
                Map<Permission, AccessExplanation> contextResult = new HashMap<>();
                AccessExplanation explanation = explainPermission(identity, groups, null, content);
                if (explanation.accessResult() != AccessResult.UNKNOWN)
                {
                    contextResult.put(new Permission(PermissionType.ALL_RIGHTS, null), explanation);
                }
                
                explanation = explainReadAccessPermission(identity, groups, content);
                if (explanation.accessResult() != AccessResult.UNKNOWN)
                {
                    contextResult.put(new Permission(PermissionType.READ, null), explanation);
                }
                
                if (!contextResult.isEmpty())
                {
                    ExplanationObject explanationContext = getExplanationObject(content);
                    result.put(explanationContext, contextResult);
                }
            }
        }
        
        return result;
    }
    
    public Map<Permission, AccessExplanation> explainAllPermissionsForAnonymous(Object object)
    {
        return Map.of();
    }
    
    public Map<Permission, AccessExplanation> explainAllPermissionsForAnyConnected(Object object)
    {
        return Map.of();
    }
    
    public Map<UserIdentity, Map<Permission, AccessExplanation>> explainAllPermissionsByUser(Object object)
    {
        UserIdentity creator = ((Content) object).getCreator();
        
        AccessResult access = _getUserPermission(creator, object);
        if (access == AccessResult.UNKNOWN)
        {
            return Map.of();
        }
        
        AccessExplanation explanation = _getAccessExplanation(access, object);
        return Map.of(
            creator,
            Map.of(
                new Permission(PermissionType.READ, null), explanation,
                new Permission(PermissionType.ALL_RIGHTS, null), explanation
            )
        );
    }
    
    public Map<GroupIdentity, Map<Permission, AccessExplanation>> explainAllPermissionsByGroup(Object object)
    {
        return Map.of();
    }
    
    public I18nizableText getObjectLabel(Object object)
    {
        if (object instanceof Content content)
        {
            return ODFContentHierarchicalAccessController.getContentObjectLabel(content, _odfContentsTreeHelper);
        }
        throw new RightsException("Unsupported context: " + object.toString());
    }

    public I18nizableText getObjectCategory(Object object)
    {
        return ODFContentHierarchicalAccessController.ODF_CONTEXT_CATEGORY;
    }
}
