/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.skill.workflow;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.cms.workflow.CreateContentFunction;
import org.ametys.cms.workflow.EditContentFunction;
import org.ametys.odf.workflow.AbstractCreateODFContentFunction;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.plugins.workflow.EnhancedFunction;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.WorkflowException;

/**
 * OSWorkflow function for creating a MacroSkill or MicroSkill content
 */
public class SkillEditionFunction extends AbstractContentWorkflowComponent implements EnhancedFunction
{
    /** The content type of macro skills */
    public static final String ASBTRACT_SKILL_TYPE = "org.ametys.plugins.odf.Content.abstractSkill";
    /** The content type of macro skills */
    public static final String MACRO_SKILL_TYPE = "org.ametys.plugins.odf.Content.macroSkill";
    /** The content type of micro skills */
    public static final String MICRO_SKILL_TYPE = "org.ametys.plugins.odf.Content.microSkill";
    /** Content name prefix for skills */
    public static final String CONTENT_NAME_PREFIX = "skill-";
    /** Constant for storing the catalog name to use into the transient variables map. */
    public static final String CONTENT_TRANSVERSAL_KEY = SkillEditionFunction.class.getName() + "$transversal";
    
    /** Constant for storing the catalog name to use into the transient variables map. */
    public static final Set<Integer> CONTENT_CREATION_ACTION_IDS = Set.of(0, 1);
    
    /** Ametys object resolver available to subclasses. */
    protected AmetysObjectResolver _resolver;
    /** The content types helper */
    protected ContentTypesHelper _contentTypesHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _contentTypesHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
    }
    
    @Override
    public void execute(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        WorkflowAwareContent content = getContent(transientVars);
        
        // If we are creating the skill, add the initial values
        if (_contentTypesHelper.isInstanceOf(content, ASBTRACT_SKILL_TYPE) && CONTENT_CREATION_ACTION_IDS.contains((int) transientVars.get("actionId")))
        {
            if (_contentTypesHelper.isInstanceOf(content, MACRO_SKILL_TYPE))
            {
                _updateMacroSkill((ModifiableContent) content, transientVars);
            }
            else if (_contentTypesHelper.isInstanceOf(content, MICRO_SKILL_TYPE))
            {
                _updateMicroSkill((ModifiableContent) content, transientVars);
            }
        }
    }
    
    private void _updateSkill(ModifiableContent skillContent, Map transientVars, Optional<String> parentId)
    {
        Optional<String> catalog = Optional.empty();
        if (parentId.isPresent())
        {
            catalog = _getParentCatalog(parentId.get());
        }
        
        if (catalog.isEmpty())
        {
            // Try to retrieve the catalog from transient vars or initial value supplier
            catalog = this.<String>_getValueFromTransientVars(transientVars, AbstractCreateODFContentFunction.CONTENT_CATALOG_KEY)
                // Used to get the catalog if the skill item is created by java code, like for the csv import.
                .or(() -> this.<String>_getValueFromInitialValueSupplier(transientVars, List.of("catalog")))
                .or(() -> this.<String>_getValueFromParameters(transientVars, "catalog"));
        }
        
        /* Set the catalog */
        if (catalog.isPresent())
        {
            // Set the catalog if value is present
            skillContent.setValue("catalog", catalog.get());
        }
        
        /* Set the code */
        
        // Used to get the code if the skill is created by java code, like for the csv import.
        _getValueFromInitialValueSupplier(transientVars, List.of("code"))
            // Set the code if value is present
            .ifPresent(value -> skillContent.setValue("code", value));
        
        // Generate a code if empty
        String code = skillContent.getValue("code");
        if (StringUtils.isEmpty(code))
        {
            skillContent.setValue("code", org.ametys.core.util.StringUtils.generateKey().toUpperCase());
        }
        
        skillContent.saveChanges();
    }
    
    private void _updateMacroSkill(ModifiableContent skillContent, Map transientVars)
    {
        
        // Try to get the parent program from parent context value transient var
        Optional<String> optionalParentId = Optional.ofNullable((String) transientVars.get(CreateContentFunction.PARENT_CONTEXT_VALUE));
    
        // Try to get the transversal attribute from initial value supplier
        Optional<Object> optionalTransversal = _getValueFromTransientVars(transientVars, CONTENT_TRANSVERSAL_KEY)
                // Used to get the transversal if the skill item is created by java code, like for the csv import.
                .or(() -> _getValueFromInitialValueSupplier(transientVars, List.of("transversal")))
                .or(() -> _getValueFromParameters(transientVars, "transversal"));
        
        /* Set the parent */
        if (optionalParentId.isPresent())
        {
            String parentId = optionalParentId.get();
            // Set the parent program if value is present and synchronize it
            skillContent.synchronizeValues(Map.of("parentProgram", parentId));
            skillContent.setValue("transversal", false);
        }
        else if (optionalTransversal.isPresent())
        {
            Object transversalObj = optionalTransversal.get();
            boolean transversal = transversalObj instanceof String ? Boolean.valueOf((String) transversalObj) : (Boolean) transversalObj;
            
            skillContent.setValue("transversal", transversal);
        }
        else
        {
            // Defaults to true (example when importing skills via CSV)
            skillContent.setValue("transversal", true);
        }
        
        _updateSkill(skillContent, transientVars, optionalParentId);
    }
    
    private void _updateMicroSkill(ModifiableContent skillContent, Map transientVars)
    {
     // Try to get the parent program from parent context value transient var
        Optional<String> optionalParentId = Optional.ofNullable((String) transientVars.get(CreateContentFunction.PARENT_CONTEXT_VALUE));
    
        /* Set the parent */
        if (optionalParentId.isPresent())
        {
            String parentId = optionalParentId.get();
            // Set the parent macro skill if value is present and synchronize it
            skillContent.synchronizeValues(Map.of("parentMacroSkill", parentId));
        }
        
        _updateSkill(skillContent, transientVars, optionalParentId);
    }
    
    private Optional<String> _getParentCatalog(String parentId)
    {
        if (StringUtils.isNotBlank(parentId))
        {
            Object parent = _resolver.resolveById(parentId);
            if (parent != null && parent instanceof Content parentContent && parentContent.hasValue("catalog"))
            {
                return Optional.of(parentContent.getValue("catalog"));
            }
        }
        
        return Optional.empty();
    }
    
    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.odf", "PLUGINS_ODF_CREATE_SKILL_CONTENT_FUNCTION_LABEL");
    }
    
    @SuppressWarnings("unchecked")
    private <T> Optional<T> _getValueFromTransientVars(Map transientVars, String attributeName)
    {
        return Optional.ofNullable((T) transientVars.get(attributeName))
                .filter(Objects::nonNull);
    }
    
    @SuppressWarnings("unchecked")
    private <T> Optional<T> _getValueFromInitialValueSupplier(Map transientVars, List<String> attributePath)
    {
        return Optional.ofNullable(transientVars.get(CreateContentFunction.INITIAL_VALUE_SUPPLIER))
                .map(Function.class::cast)
                .map(function -> function.apply(attributePath))
                .filter(Objects::nonNull)
                .map(value -> (T) value);
    }
    
    @SuppressWarnings("unchecked")
    private <T> Optional<T> _getValueFromParameters(Map transientVars, String attributeName)
    {
        Object objectParameters = transientVars.get(AbstractWorkflowComponent.CONTEXT_PARAMETERS_KEY);
        
        if (objectParameters != null && objectParameters instanceof Map parameters)
        {
            Object values = parameters.get(EditContentFunction.FORM_RAW_VALUES);
            if (values != null && values instanceof Map valuesMap)
            {
                return Optional.ofNullable((T) valuesMap.get(EditContentFunction.FORM_ELEMENTS_PREFIX + attributeName))
                        .filter(Objects::nonNull);
            }
        }
        
        return Optional.empty();
    }
}
