/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.container;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.content.CopyODFContentUpdater;
import org.ametys.odf.program.Container;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.WorkflowException;

/**
 * Copy updater to remove the 'previousYears' and 'nexYears' attributes on {@link Container}.
 */
public class ContainerCopyUpdater extends AbstractLogEnabled implements CopyODFContentUpdater, Serviceable
{
    private ODFHelper _odfHelper;
    private ContentWorkflowHelper _contentWorkflowHelper;

    public void service(ServiceManager smanager) throws ServiceException
    {
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) smanager.lookup(ContentWorkflowHelper.ROLE);
    }

    @Override
    public void updateContents(String initialCatalogName, String newCatalogName, Map<Content, Content> copiedContents, Content targetParentContent)
    {
        for (Content copiedContent : copiedContents.values())
        {
            if (copiedContent instanceof Container copiedContainer && _odfHelper.isContainerOfTypeYear(copiedContainer))
            {
                try
                {
                    // Remove previousYears, nextYears and pathways references (invert relations should be updated too)
                    ContentValue[] refPreviousYears = copiedContainer.getValue("previousYears");
                    ContentValue[] refNextYears = copiedContainer.getValue("nextYears");
                    ContentValue[] pathways = copiedContainer.getValue("pathways");
                    
                    if (refPreviousYears != null && refPreviousYears.length > 0 || refNextYears != null && refNextYears.length > 0 || pathways != null && pathways.length > 0)
                    {
                        _contentWorkflowHelper.editContent(copiedContainer, Map.of("nextYears", List.of(), "previousYears", List.of(), "pathways", List.of()), 2);
                    }
                }
                catch (WorkflowException e)
                {
                    getLogger().error("Unable to remove 'previousYears' and 'nextYears' references from copied year container ''{0}''.", e, copiedContainer.getName());
                }
            }
        }
    }
}
