/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.inputdata;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashSet;
import java.util.Set;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfiguration;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.core.util.IgnoreRootHandler;
import org.ametys.runtime.plugin.ExtensionPoint;
import org.ametys.runtime.plugin.component.AbstractThreadSafeComponentExtensionPoint;

/**
 * This {@link ExtensionPoint} handles the pool of available {@link ContentInputData}.
 */
public class ContentInputDataExtensionPoint extends AbstractThreadSafeComponentExtensionPoint<ContentInputData> implements Configurable
{
    /** Avalon role */
    public static final String ROLE = ContentInputDataExtensionPoint.class.getName();
    
    private SourceResolver _resolver;
    
    private Set<String> _inputDataRoles;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (SourceResolver) serviceManager.lookup(SourceResolver.ROLE);
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _inputDataRoles = new HashSet<>();
        
        // Replace with the default configuration variable when RUNTIME-611 is implemented.
        Configuration conf = getExternalConfiguration();
        
        for (Configuration inputDataConfiguration : conf.getChildren("content-input-data"))
        {
            String role = inputDataConfiguration.getValue(null);
            _inputDataRoles.add(role);
        }
    }
    
    @Override
    public void initializeExtensions() throws Exception
    {
        super.initializeExtensions();
        
        for (String inputDataRole : _inputDataRoles)
        {
            ContentInputData inputData = getExtension(inputDataRole);
            
            if (inputData == null)
            {
                throw new RuntimeException("Unknown ROLE for content input data: " + inputDataRole);
            }
        }
    }
    
    /**
     * SAX all the input data.
     * @param handler the content handler to SAX into.
     * @throws SAXException if an error occurs while SAXing.
     * @throws ProcessingException if an error occurs while processing.
     */
    public void toSAX(ContentHandler handler) throws SAXException, ProcessingException
    {
        for (String inputDataRole : _inputDataRoles)
        {
            ContentInputData inputData = getExtension(inputDataRole);
            
            inputData.toSAX(new IgnoreRootHandler(handler));
        }
    }
    
    /**
     * Get the component configuration as a Configuration object.
     * @return the component Configuration.
     * @throws ConfigurationException if an error occurs
     */
    protected Configuration getExternalConfiguration() throws ConfigurationException
    {
        String configFileName = "context://WEB-INF/param/inputdata-content.xml";
        
        Configuration conf = new DefaultConfiguration("input-data-collection");
        
        Source configSource = null;
        
        try 
        {
            configSource = _resolver.resolveURI(configFileName);
            if (configSource.exists())
            {
                try (InputStream configStream = configSource.getInputStream())
                {
                    conf = new DefaultConfigurationBuilder().build(configStream);
                }
            }
            return conf;
        }
        catch (IOException e)
        {
            throw new ConfigurationException("Error trying to read the content input data configuration file.", e);
        }
        catch (SAXException e)
        {
            throw new ConfigurationException("Error trying to read the content input data configuration file.", e);
        }
        finally
        {
            _resolver.release(configSource);
        }
    }
}
