/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Singleton class to export a catalog to InDesign format.
 * @private
 */
Ext.define('Ametys.plugins.odf.indesign.ExportCatalogInDesign', {
	singleton: true,
	
	/**
 	 * @private
 	 * @property {Ametys.window.DialogBox} _box The dialog box for creating/editing a catalog.
 	 */
 	 
 	/**
 	 * @private
 	 * @property {Ext.form.Panel} _formPanel The form panel of the dialog box.
 	 */
 	
 	/**
	 * @private
	 * @property {Boolean} _initialized Indicates if the create/edit dialog box is initialized.
	 */
	
	/**
	 * Exports the catalog in InDesign format.
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
	 */
	act: function (controller)
	{
		this.open(this._doExport);
	},
    
    /**
     * Open the dialog box to prepare InDesign export
     * @param {Function} callback The callback function responsible for effective export
     * @param {Object} callback.values The chosen export parameters
     * @param {String} callback.values.catalog The selected catalog
     * @param {String} callback.values.xslt The XSLT file for unit export
     * @param {String} callback.values.globalXslt The XSLT file for global export
     */
    open: function (callback)
    {
        this._cbFn = callback;
        this._delayedInitialize();
        this._initForm();
        this._box.show();
    },
    
    /**
     * @private
     * Do export
     * @param {Object} values The selected values
     * @param {String} values.catalog The selected catalog
     * @param {String} values.xslt The XSLT file for unit export
     * @param {String} values.globalXslt The XSLT file for global export
     */
    _doExport: function (values)
    {
        Ametys.openWindow(Ametys.getPluginDirectPrefix('odf') + '/indesign/' + values.catalog + '/catalogue.xml', values);
    },
	
	/**
	 * @private
	 * Creates the dialog box
	 */
	_delayedInitialize: function ()
	{
		if (!this._initialized)
		{
			this._formPanel = this._createFormPanel();
			
			this._box = Ext.create('Ametys.window.DialogBox', {
				title: "{{i18n PLUGINS_ODF_EXPORT_INDESIGN_DIALOGBOX_TITLE}}",
	            icon: Ametys.getPluginResourcesPrefix('odf') + '/img/actions/export_indesign_16.png',
	            
	            width: 450,
		        scrollable: false,
	            
	            items: [this._formPanel],
	            
	            defaultFocus: 'title',
	            closeAction: 'hide',
	            buttons: [{
	                text: "{{i18n PLUGINS_ODF_EXPORT_INDESIGN_DIALOGBOX_OK}}",
	                handler: Ext.bind(this._validate, this)
	            }, {
	                text: "{{i18n PLUGINS_ODF_EXPORT_INDESIGN_DIALOGBOX_CANCEL}}",
	                handler: Ext.bind( function() {this._box.close();}, this)
	            }]    
			});
			
			this._initialized = true;
		}
	},
	
	/**
 	 * Creates the form panel of this dialog box.
 	 * @return {Ext.form.Panel} The form panel
 	 * @private
 	 */
 	_createFormPanel: function()
 	{
 		var catalogFd = Ametys.odf.catalog.CatalogDAO.createComboBox({
            name: 'catalog',
            itemId: 'catalog',
            allowBlank: false
        });
		
		var formPanel = Ext.create('Ext.form.Panel', {
	        defaultType: 'combo',
	        defaults: {
	        	cls: 'ametys',
	        	labelSeparator: '',
	        	labelAlign: 'right',
	        	labelWidth: 120,
	        	width: '100%',
	        	msgTarget: 'side'
	        },
	        
	        border: false,
	        scrollable: true,
	        
	        items: [
		        {
		        	xtype: 'container',
		        	itemId: 'hint',
		        	html: "{{i18n PLUGINS_ODF_EXPORT_CATALOG_INDESIGN_DIALOGBOX_HINT}}",
		        	cls: 'a-text'
		        },
                catalogFd,
		        {
		        	name: 'xslt',
		        	itemId: 'xslt',
		        	fieldLabel: "{{i18n PLUGINS_ODF_EXPORT_CATALOG_INDESIGN_DIALOGBOX_XSLT_SINGLE}}",
	                
	                store: {
	                	fields: [
				        	{name: 'value', mapping: 'name'},
				        	{name: 'displayText', mapping: 'label', type: 'string'}
				    	],
				    	sorters: {property: 'displayText', direction:'ASC'},
				    	proxy: {
				    		type: 'ametys',
				    		plugin: 'odf',
							url: 'indesign/get-xslt.json',
				    		reader: {
				    			type: 'json',
				    			rootProperty: 'files'
				    		}
				    	},
				    	autoLoad: true
	                },
	                valueField: 'value',
	                displayField: 'displayText',
	                
	                queryMode: 'local',
	                allowBlank: false,
	                forceSelection : true,
	                triggerAction: 'all'
		        },
		        {
		        	name: 'globalXslt',
		        	itemId: 'globalXslt',
		        	fieldLabel: "{{i18n PLUGINS_ODF_EXPORT_CATALOG_INDESIGN_DIALOGBOX_XSLT_GLOBAL}}",
	                
	                store: {
	                	fields: [
				        	{name: 'value', mapping: 'name'},
				        	{name: 'displayText', mapping: 'label', type: 'string'}
				    	],
				    	sorters: {property: 'displayText', direction:'ASC'},
				    	proxy: {
				    		type: 'ametys',
				    		plugin: 'odf',
							url: 'indesign/catalog/get-xslt.json',
				    		reader: {
				    			type: 'json',
				    			rootProperty: 'files'
				    		}
				    	},
				    	autoLoad: true
	                },
	                valueField: 'value',
	                displayField: 'displayText',
	                
	                queryMode: 'local',
	                allowBlank: false,
	                forceSelection : true,
	                triggerAction: 'all'
		        }
	        ]
        	
 		});
 		
 		return formPanel;
 	},
 	
 	/**
 	 * @private
 	 * Initializes the form with some optional values.
 	 */
	_initForm: function ()
 	{
 		var form = this._formPanel.getForm();
		form.reset();
        
        Ametys.odf.catalog.CatalogDAO.getDefaultCatalogName([], function(catalogName) {
            form.findField('catalog').setValue(catalogName)
        }, this);
 	},
 	
 	/**
 	 * @private
 	 * Handler for the 'ok' button of the dialog box
 	 */ 
    _validate: function ()
    {
        var form = this._formPanel.getForm();
        if (!form.isValid())
        {
        	return;
        }
        
        var values = form.getValues();

        if (Ext.isFunction(this._cbFn))
        {
            this._cbFn(values);
        }
        
		this._box.close();
    }
	
});