/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.cdmfr;

import java.io.IOException;
import java.io.InputStream;
import java.util.Set;
import java.util.stream.Stream;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;

import org.ametys.odf.ODFHelper;
import org.ametys.odf.program.Program;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Processor to process CDM-fr files.
 *  - Filter programs if need
 *  - Iterate on programs
 *  - Generate the CDM-fr file
 *  - Process on the CDM-fr file
 */
public abstract class AbstractCDMfrProcessor extends AbstractLogEnabled implements Component, Serviceable, Initializable
{
    /** The source resolver */
    protected SourceResolver _sourceResolver;

    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    private boolean _isActive;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
    }
    
    public void initialize() throws Exception
    {
        _isActive = Config.getInstance().getValue(getConfigActiveParam());
    }
    
    /**
     * Get the configuration parameter name to determine if the process is active.
     * @return The configuration parameter name
     */
    protected abstract String getConfigActiveParam();
    
    /**
     * <code>true</code> if the process is active
     * @return <code>true</code> if the process is active
     */
    public boolean isActive()
    {
        return _isActive;
    }
    
    /**
     * Process the CDM-fr of the given programs.
     * @param programs The programs to process
     */
    public void processPrograms(Set<Program> programs)
    {
        // Check if the process is active
        if (isActive())
        {
            // Filter programs if needed and iterate on it
            filterPrograms(programs.stream()).forEach(
                program ->
                {
                    // Get the CDM-fr content and process the program
                    Source cdmfrSource = null;
                    try
                    {
                        cdmfrSource = _getCDMfrContent(program);
                        try (InputStream is = cdmfrSource != null ? cdmfrSource.getInputStream() : null)
                        {
                            processProgram(program, is);
                        }
                    }
                    catch (IOException e)
                    {
                        getLogger().error(getErrorMessage(), e);
                    }
                    finally
                    {
                        if (cdmfrSource != null)
                        {
                            _sourceResolver.release(cdmfrSource);
                        }
                    }
                }
            );
        }
    }
    
    /**
     * Process the program and its CDM-fr input stream.
     * @param program The program
     * @param cdmfrContent The CDM-fr content as {@link InputStream}
     * @throws IOException if an exception occurs
     */
    protected abstract void processProgram(Program program, InputStream cdmfrContent) throws IOException;
    
    /**
     * Get the error message if process fails.
     * @return the error message
     */
    protected abstract String getErrorMessage();
    
    /**
     * Filter programs if needed.
     * @param programs The programs to filter
     * @return The filtered programs
     */
    protected Stream<Program> filterPrograms(Stream<Program> programs)
    {
        // Default implementation does not filter anything
        return programs;
    }
    
    /**
     * Get the CDM-fr of the current program.
     * @param program The program to export as CDM-fr
     * @return The {@link InputStream} with the CDM-fr content
     * @throws IOException if an exception occurs
     */
    protected Source _getCDMfrContent(Program program) throws IOException
    {
        // Build the URI to export program as CDM-fr
        StringBuilder sb = new StringBuilder("cocoon://_plugins/odf/export-cdmfr.xml");
        sb.append("?id=").append(program.getId());
        sb.append("&").append(ExportCDMfrHelper.REQUEST_PARAM_VALID_LABEL).append("=true");
        if (isCDMfrForAmetys())
        {
            sb.append("&").append(ExportCDMfrHelper.REQUEST_PARAM_EXPORT_FOR_AMETYS).append("=true");
        }
        
        // Generate CDM-fr file
        return _sourceResolver.resolveURI(sb.toString());
    }
    
    /**
     * <code>true</code> to generate CDM-fr for another Ametys, <code>false</code> otherwise.
     * @return <code>true</code> to generate CDM-fr for another Ametys instance
     */
    protected abstract boolean isCDMfrForAmetys();
}
