/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.observation;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.content.indexing.solr.SolrIndexer;
import org.ametys.cms.content.indexing.solr.observation.ObserverHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.AsyncObserver;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;


/**
 * {@link Observer} when a educational path is no more valid after moving or removing a program item in ODF tree
 */
public class IndexSharedPropertyOnHierarchyChangedObserver extends AbstractLogEnabled implements AsyncObserver, Serviceable
{
    private AmetysObjectResolver _resolver;
    private SolrIndexer _solrIndexer;
    private ODFHelper _odfHelper;

    public void service(ServiceManager smanager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _solrIndexer = (SolrIndexer) smanager.lookup(SolrIndexer.ROLE);
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
    }

    public boolean supports(Event event)
    {
        return event.getId().equals(OdfObservationConstants.EVENT_PROGRAM_ITEM_HIERARCHY_CHANGED);
    }

    public int getPriority()
    {
        return MIN_PRIORITY;
    }

    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        if (ObserverHelper.isNotSuspendedObservationForIndexation())
        {
            ProgramItem programItem = getProgramItem(event);
            if (programItem != null)
            {
                _indexSharedProperty(programItem);
            }
        }
    }
    
    private void _indexSharedProperty(ProgramItem programItem) throws Exception
    {
        // Do a partial update of the shared property
        _solrIndexer.updateProperty((Content) programItem, ProgramItem.SHARED_PROPERTY, RepositoryConstants.DEFAULT_WORKSPACE);
        
        List<ProgramItem> childProgramItems = _odfHelper.getChildProgramItems(programItem);
        for (ProgramItem childProgramItem : childProgramItems)
        {
            _indexSharedProperty(childProgramItem);
        }
    }
    
    /**
     * Get the {@link ProgramItem} concerned by this event
     * @param event the event
     * @return the program item content or null
     */
    protected ProgramItem getProgramItem(Event event)
    {
        Map<String, Object> args = event.getArguments();

        if (args.containsKey(OdfObservationConstants.ARGS_PROGRAM_ITEM_ID))
        {
            String programItemId = (String) args.get(OdfObservationConstants.ARGS_PROGRAM_ITEM_ID);
            return _resolver.resolveById(programItemId);
        }
        return null;
    }
}
