/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.workflow;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang.StringUtils;

import org.ametys.cms.repository.ModifiableWorkflowAwareContent;
import org.ametys.cms.workflow.CreateContentFunction;
import org.ametys.odf.ODFHelper;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.collection.AmetysObjectCollection;
import org.ametys.runtime.config.Config;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.WorkflowException;

/**
 * Abstract OSWorkflow function for creating a ODF content.
 */
public abstract class AbstractCreateODFContentFunction extends CreateContentFunction
{
    /** Constant for storing the catalog name to use into the transient variables map. */
    public static final String CONTENT_CATALOG_KEY = AbstractCreateODFContentFunction.class.getName() + "$catalog";
    
    /** ODF helper */
    protected ODFHelper _odfHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
    }
    
    @Override
    protected AmetysObjectCollection< ? , ModifiableWorkflowAwareContent> _getContentRoot(Map transientVars) throws WorkflowException
    {
        return _odfHelper.getRootContent(true);
    }
    
    @Override
    public void execute(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        if (StringUtils.isEmpty((String) transientVars.get(CONTENT_LANGUAGE_KEY)))
        {
            // Override the content language with the odf main language
            String odfLang = Config.getInstance().getValue("odf.programs.lang");
            transientVars.put(CONTENT_LANGUAGE_KEY, odfLang);
        }
        
        super.execute(transientVars, args, ps);
    }
    
    @Override
    protected ModifiableWorkflowAwareContent _createContent(Map transientVars, Map args, String desiredContentName, ModifiableTraversableAmetysObject contentsNode)
    {
        String baseName = desiredContentName;
        String prefix = _getContentNamePrefix();
        if (!baseName.startsWith(prefix))
        {
            baseName = prefix + baseName;
        }
        
        return super._createContent(transientVars, args, baseName, contentsNode);
    }
    
    /**
     * Get the prefix for content name
     * @return the prefix
     */
    protected abstract String _getContentNamePrefix ();
}
