/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.content.code;

import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.model.properties.AbstractIndexableStaticProperty;
import org.ametys.cms.model.properties.Property;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ProgramItem;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * {@link Property} for the computed display code of a {@link ProgramItem}.
 */
public class DisplayCodeProperty extends AbstractIndexableStaticProperty<String, String, Content>
{
    /** The property name for display code */
    public static final String PROPERTY_NAME = "displayCode";
    
    private static final String __CONFIG_PARAM_DISPLAY_CODE_PROVIDER = "odf.display.code.provider";
    
    private DisplayCodeProviderExtensionPoint _displayCodeEP;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _displayCodeEP = (DisplayCodeProviderExtensionPoint) smanager.lookup(DisplayCodeProviderExtensionPoint.ROLE);
    }
    
    public Object getValue(Content content)
    {
        String dcpId = Config.getInstance().getValue(__CONFIG_PARAM_DISPLAY_CODE_PROVIDER, true, AmetysDisplayCodeProvider.ID);
        
        DisplayCodeProvider dcProvider = _displayCodeEP.getExtension(dcpId);
        if (dcProvider.supports(content))
        {
            Optional<String> displayCode = dcProvider.getDisplayCode(content);
            if (displayCode.isPresent())
            {
                return displayCode.get();
            }
        }
        
        // Return value of default provider
        DisplayCodeProvider defaultProvider = _displayCodeEP.getExtension(AmetysDisplayCodeProvider.ID);
        return defaultProvider.getDisplayCode(content).orElse(null);
    }

    @Override
    protected String getTypeId()
    {
        return ModelItemTypeConstants.STRING_TYPE_ID;
    }
}
