/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.schedulable;

import java.util.ArrayList;
import java.util.List;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.quartz.JobExecutionContext;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentDAO;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.odf.skill.workflow.SkillEditionFunction;
import org.ametys.plugins.core.impl.schedule.AbstractStaticSchedulable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.BooleanExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.MetadataExpression;
import org.ametys.plugins.repository.query.expression.NotExpression;

/**
 * Scheduler to purge orphan skills.
 */
public class SkillsPurgeSchedulable extends AbstractStaticSchedulable
{
    private AmetysObjectResolver _resolver;
    private ContentDAO _contentDAO;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _contentDAO = (ContentDAO) manager.lookup(ContentDAO.ROLE);
    }
    
    @Override
    public void execute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        List<Content> contentIdsToDelete = new ArrayList<>();
        
        contentIdsToDelete.addAll(_getOrphansMacroSkills());
        contentIdsToDelete.addAll(_getOrphansMicroSkills());
        
        // Remove the skills
        _contentDAO.forceDeleteContentsObj(contentIdsToDelete, null);
    }
    
    private List<Content> _getOrphansSkills(Expression expression)
    {
        String query = ContentQueryHelper.getContentXPathQuery(expression);
        
        return _resolver.<Content>query(query).stream()
                // Log
                .peek(skill -> getLogger().info("The purge will delete the content {}.", skill.getId().toString()))
                .toList();
    }
    
    private List<Content> _getOrphansMacroSkills()
    {
        AndExpression expression = new AndExpression();
        expression.add(new ContentTypeExpression(Operator.EQ, SkillEditionFunction.MACRO_SKILL_TYPE));
        expression.add(new NotExpression(new MetadataExpression("parentProgram")));
        
        // The orphans macroskills are the one that don't have a parent and that are also not transversal
        expression.add(new BooleanExpression("transversal", false));
        
        return _getOrphansSkills(expression);
    }
    
    private List<Content> _getOrphansMicroSkills()
    {
        AndExpression expression = new AndExpression();
        expression.add(new ContentTypeExpression(Operator.EQ, SkillEditionFunction.MICRO_SKILL_TYPE));
        expression.add(new NotExpression(new MetadataExpression("parentMacroSkill")));
        
        return _getOrphansSkills(expression);
    }
}
