/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.workflow;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.type.impl.BooleanRepositoryElementType;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.workflow.EditContentFunction;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.odf.EducationalPathHelper;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.rights.ODFRightHelper.ContextualizedContent;
import org.ametys.plugins.repository.data.holder.values.SynchronizableRepeater;
import org.ametys.plugins.repository.data.holder.values.SynchronizableValue;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.plugins.repository.model.ViewHelper;
import org.ametys.runtime.model.ModelHelper;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelItemGroup;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.ViewItemContainer;
import org.ametys.runtime.model.disableconditions.AbstractRelativeDisableCondition;
import org.ametys.runtime.model.disableconditions.DisableCondition;
import org.ametys.runtime.model.disableconditions.DisableConditions;
import org.ametys.runtime.model.type.DataContext;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.WorkflowException;

/**
 * Edit content function restricted to contextualized data, ie. data into repeater with educational path
 * This function allows to edit data by educational path for which the user is responsible.
 */
public class EditContextualizedDataFunction extends EditContentFunction
{
    /** The workflow action id to edit repeater with educational path only */
    public static final int EDIT_WORKFLOW_ACTION_ID = 20;
    
    /** Name of educational path attribute in a repeater with education path */
    public static final String REPEATER_EDUCATIONAL_PATH_ATTRIBUTE_NAME = "path";
    /** Name of common attribute in a repeater with education path (means that entry is common to all educational paths)*/
    public static final String REPEATER_COMMON_ATTRIBUTE_NAME = "common";
    
    private EducationalPathHelper _educationalPathHelper;
    private RightManager _rightManager;
    private String _rightId;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _educationalPathHelper = (EducationalPathHelper) smanager.lookup(EducationalPathHelper.ROLE);
        _rightManager = (RightManager) smanager.lookup(RightManager.ROLE);
    }
    
    @Override
    public void execute(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        _rightId = (String) args.get("right");
        super.execute(transientVars, args, ps);
    }
    
    @Override
    protected boolean canWriteModelItem(ModelItem modelItem, Content content, Map transientVars)
    {
        // Only repeater with educational path and its child model items are writable
        // Always enable boolean "common" item into a repeater with educational path (whatever model item's restrictions)
        return _inRepeaterWithEducationalPath(modelItem) && (_isCommonBooleanItem(modelItem) || super.canWriteModelItem(modelItem, content, transientVars));
    }
    
    private boolean _isCommonBooleanItem(ModelItem modelItem)
    {
        return modelItem.getType() instanceof BooleanRepositoryElementType
                && modelItem.getName().equals(REPEATER_COMMON_ATTRIBUTE_NAME)
                && modelItem.getParent() != null
                && modelItem.getParent() instanceof RepeaterDefinition;
    }
    
    private boolean _inRepeaterWithEducationalPath(ModelItem modelItem)
    {
        if (modelItem instanceof RepeaterDefinition repeaterDefinition && _educationalPathHelper.isRepeaterWithEducationalPath(repeaterDefinition))
        {
            return true;
        }
        
        ModelItemGroup parentModelItem = modelItem.getParent();
        return parentModelItem != null ? _inRepeaterWithEducationalPath(parentModelItem) : false;
    }
    
    @Override
    protected Map<String, Object> parseValues(View view, ModifiableContent content, Map<String, Object> rawValues, boolean localOnly, Map transientVars)
    {
        Map<String, Object> values = super.parseValues(view, content, rawValues, localOnly, transientVars);
        
        _filterAndUpdateRepeaterWithPathValues(content, view, values);
        
        return values;
    }
    
    @Override
    protected Map<String, Object> convertValues(ModifiableContent content, View view, Map<String, Object> values, Map transientVars)
    {
        Map<String, Object> convertValues = super.convertValues(content, view, values, transientVars);
        
        _filterAndUpdateRepeaterWithPathValues(content, view, convertValues);
        
        return convertValues;
    }

    @SuppressWarnings("unchecked")
    private void _filterAndUpdateRepeaterWithPathValues(Content content, ViewItemContainer viewItemContainer, Map<String, Object> values)
    {
        ViewHelper.visitView(viewItemContainer,
            (element, definition) -> {
                // nothing
            },
            (group, definition) -> {
                // composite
                String name = definition.getName();
                Map<String, Object> composite = (Map<String, Object>) values.get(name);
                if (composite != null)
                {
                    _filterAndUpdateRepeaterWithPathValues(content, group, composite);
                }
            },
            (group, definition) -> {
                // repeater
                String name = definition.getName();
                
                SynchronizableRepeater repeaterValue = (SynchronizableRepeater) values.get(name);
                List<Map<String, Object>> entries = ((SynchronizableRepeater) values.get(name)).getEntries();
                if (entries != null)
                {
                    if (_educationalPathHelper.isRepeaterWithEducationalPath(definition))
                    {
                        List<Map<String, Object>> storedEntries = _getRepeaterEntries(content, definition);
                        values.put(name, _updateRepeaterWithPathValues(content, repeaterValue, entries, storedEntries, definition));
                    }
                    else
                    {
                        for (Map<String, Object> entry : entries)
                        {
                            _filterAndUpdateRepeaterWithPathValues(content, group, entry);
                        }
                    }
                }
            },
            group -> _filterAndUpdateRepeaterWithPathValues(content, group, values));
    }
    
    // Update repeater values recursively. Only entries with allowed path will be really updated
    private List<Map<String, Object>> _updateRepeaterWithPathValues(Content content, SynchronizableRepeater repeaterValue, List<Map<String, Object>> entryValues, List<Map<String, Object>> storedEntries, RepeaterDefinition repeaterDefinition)
    {
        List<Map<String, Object>> updatedValues = new ArrayList<>();
        
        if (storedEntries != null && !storedEntries.isEmpty())
        {
            // Get common entries and entries with unauthorized path from content
            int entryPos = 1;
            for (Map<String, Object> storedEntry : storedEntries)
            {
                boolean commonEntry = _isCommonEntry(repeaterDefinition, storedEntry);
                EducationalPath educationalPath = (EducationalPath) storedEntry.get(REPEATER_EDUCATIONAL_PATH_ATTRIBUTE_NAME);
                
                if (commonEntry || educationalPath == null)
                {
                    // Get the names of sub repeaters if exists
                    List<String> subRepeaterNames = repeaterDefinition
                            .getModelItems()
                            .stream()
                            .filter(i -> i instanceof RepeaterDefinition)
                            .map(ModelItem::getName)
                            .toList();
                    
                    // Get values from stored entry (user not authorized to update them), except from sub-repeaters
                    Map<String, Object> updatedEntryValues = storedEntry
                        .entrySet()
                        .stream()
                        .filter(e -> !subRepeaterNames.contains(e.getKey()))
                        .collect(Collectors.toMap(e -> e.getKey(), e -> e.getValue()));
                    
                    // Get repeater entry values from client side
                    int newPos = repeaterValue.getPositionsMapping().get(entryPos);
                    Map<String, Object> entryValue = entryValues.get(newPos - 1);
                    
                    // Update sub-repeater values recursively
                    for (String subRepeaterName : subRepeaterNames)
                    {
                        RepeaterDefinition subRepeaterDefinition = (RepeaterDefinition) repeaterDefinition.getModelItem(subRepeaterName);
                        SynchronizableRepeater subRepeaterValue = (SynchronizableRepeater) entryValue.get(subRepeaterName);
                        
                        @SuppressWarnings("unchecked")
                        List<Map<String, Object>> storedSubEntries = (List<Map<String, Object>>) storedEntry.get(subRepeaterName);
                        
                        updatedEntryValues.put(subRepeaterName, _updateRepeaterWithPathValues(content, subRepeaterValue, subRepeaterValue.getEntries(), storedSubEntries, subRepeaterDefinition));
                    }
                    
                    updatedValues.add(updatedEntryValues);
                }
                else if (_rightManager.currentUserHasRight(_rightId, new ContextualizedContent(content, educationalPath)) != RightResult.RIGHT_ALLOW)
                {
                    // User is not allowed to update this entry => get value from storage
                    updatedValues.add(storedEntry);
                }
                
                entryPos++;
            }
        }
        
        // Get from client only entries with authorized path and force common attribute to false
        for (Map<String, Object> entryValue : entryValues)
        {
            if (entryValue.containsKey(REPEATER_EDUCATIONAL_PATH_ATTRIBUTE_NAME))
            {
                boolean commonEntry = _isCommonEntry(repeaterDefinition, entryValue);
                Object value = entryValue.get(REPEATER_EDUCATIONAL_PATH_ATTRIBUTE_NAME);
                value = value instanceof SynchronizableValue synchronizableValue ? synchronizableValue.getLocalValue() : value;
                
                if (!commonEntry && value instanceof EducationalPath educationalPath)
                {
                    if (_rightManager.currentUserHasRight(_rightId, new ContextualizedContent(content, educationalPath)) == RightResult.RIGHT_ALLOW)
                    {
                        if (repeaterDefinition.hasModelItem(REPEATER_COMMON_ATTRIBUTE_NAME))
                        {
                            entryValue.put(REPEATER_COMMON_ATTRIBUTE_NAME, false); // Force common to false (required to pass disable condition)
                        }
                        updatedValues.add(entryValue);
                    }
                }
            }
        }
        
        return updatedValues;
    }
    
    private boolean _isCommonEntry(RepeaterDefinition repeaterDefinition, Map<String, Object> entryValues)
    {
        boolean commonEntry = repeaterDefinition.hasModelItem(REPEATER_COMMON_ATTRIBUTE_NAME); // default true if repeater supports common entries
        if (entryValues.containsKey(REPEATER_COMMON_ATTRIBUTE_NAME))
        {
            Object commonValue = entryValues.get(REPEATER_COMMON_ATTRIBUTE_NAME);
            commonValue = commonValue instanceof SynchronizableValue synchronizableValue ? synchronizableValue.getLocalValue() : commonValue;
            if (commonValue != null)
            {
                commonEntry = (boolean) commonValue;
            }
        }
        return commonEntry;
    }
    
    private List<Map<String, Object>> _getRepeaterEntries(Content content, RepeaterDefinition repeaterDefinition)
    {
        List<String> itemPaths = new ArrayList<>();
        itemPaths.add(repeaterDefinition.getPath());
        
        // Add items needed to evaluate disabled conditions
        itemPaths.addAll(_getDisabledConditionPaths(repeaterDefinition));
        
        View view = View.of(content.getModel(), itemPaths.toArray(String[]::new));
        
        Map<String, Object> dataToMap = content.dataToMap(view, DataContext.newInstance().withDisabledValues(true).withEmptyValues(false));
        
        return _getRepeaterEntries(dataToMap, repeaterDefinition.getPath());
    }
    
    @SuppressWarnings("unchecked")
    private List<Map<String, Object>> _getRepeaterEntries(Map<String, Object> data, String path)
    {
        String[] pathSegments = StringUtils.split(path, ModelItem.ITEM_PATH_SEPARATOR);
        if (pathSegments.length == 1)
        {
            return (List<Map<String, Object>>) data.get(pathSegments[0]);
        }
        else
        {
            Map<String, Object> subData = (Map<String, Object>) data.get(pathSegments[0]);
            String remainPath = StringUtils.join(pathSegments, ModelItem.ITEM_PATH_SEPARATOR, 1, pathSegments.length);
            return _getRepeaterEntries(subData, remainPath);
        }
    }
    
    private Set<String> _getDisabledConditionPaths(ModelItem modelItem)
    {
        DisableConditions<? extends DisableCondition> disableConditions = modelItem.getDisableConditions();
        if (disableConditions != null)
        {
            Set<String> disableConditionPaths = disableConditions.getConditions()
                    .stream()
                    .filter(AbstractRelativeDisableCondition.class::isInstance)
                    .map(c -> ModelHelper.getDisableConditionAbsolutePath(c, modelItem.getPath()))
                    .collect(Collectors.toSet());
            
            if (modelItem instanceof ModelItemGroup group)
            {
                for (ModelItem childModelItem : group.getChildren())
                {
                    disableConditionPaths.addAll(_getDisabledConditionPaths(childModelItem));
                }
            }
            
            return disableConditionPaths;
        }
        return Set.of();
    }
}
