/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.course;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.ContentDataHelper;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.core.user.UserIdentity;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.course.ShareableCourseStatusHelper.ShareableStatus;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper for shareable course
 */
public class ShareableCourseHelper extends AbstractLogEnabled implements Component, Serviceable, Configurable
{
    /** The component role. */
    public static final String ROLE = ShareableCourseHelper.class.getName();

    private static final String __OWN_KEY = "OWN";
    
    /** The shareable course configuration */
    protected ShareableConfiguration _shareableCourseConfiguration;
    
    /** The ODF reference table helper */
    protected OdfReferenceTableHelper _odfRefTableHelper;
    
    /** The shareable course status */
    protected ShareableCourseStatusHelper _shareableCourseStatus;
    
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _odfRefTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _shareableCourseStatus = (ShareableCourseStatusHelper) manager.lookup(ShareableCourseStatusHelper.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public void configure(Configuration configuration) throws ConfigurationException
    {
        if ("shareable-fields".equals(configuration.getName()))
        {
            _shareableCourseConfiguration = _parseConfiguration(configuration);
        }
        else
        {
            _shareableCourseConfiguration = _parseConfiguration(configuration.getChild("shareable-fields"));
        }
        
    }
    
    /**
     * Parse configuration to create a shareable course configuration
     * @param configuration the configuration
     * @return the shareable course configuration
     */
    protected ShareableConfiguration _parseConfiguration(Configuration configuration)
    {
        boolean autoValidated = _parseAutoValidated(configuration);
        
        ShareableField programField = _parseProgramField(configuration);
        ShareableField degreeField = _parseDegreeField(configuration);
        ShareableField periodField = _parsePeriodField(configuration);
        ShareableField orgUnitField = _parseOrgUnitField(configuration);
        
        return new ShareableConfiguration(programField, degreeField, orgUnitField, periodField, autoValidated);
    }

    
    /**
     * Parse configuration to know if courses are validated after initialization 
     * @param configuration the configuration
     * @return <code>true</code> if courses are validated after initialization 
     */
    protected boolean _parseAutoValidated(Configuration configuration)
    {
        return Boolean.valueOf(configuration.getAttribute("auto-validated", "false"));
    }
    
    /**
     * Parse configuration to get program field 
     * @param configuration the configuration
     * @return the program field
     */
    protected ShareableField _parseProgramField(Configuration configuration)
    {
        Configuration programsConf = configuration.getChild(ShareableCourseConstants.PROGRAMS_FIELD_ATTRIBUTE_NAME);
        String programValuesAsString = programsConf.getValue(StringUtils.EMPTY);
        if (__OWN_KEY.equals(programValuesAsString))
        {
            return new ShareableField(true, Set.of());
        }
        else
        {
            List<String> valuesAsList = Arrays.asList(StringUtils.split(programValuesAsString, ","));
            Set<String> programValues = valuesAsList.stream()
                .map(StringUtils::trim)
                .filter(StringUtils::isNotBlank)
                .filter(this::_isContentExist)
                .collect(Collectors.toSet());
            
            return new ShareableField(false, programValues);
        }
    }
    
    /**
     * Parse configuration to get degree field 
     * @param configuration the configuration
     * @return the degree field
     */
    protected ShareableField _parseDegreeField(Configuration configuration)
    {
        Configuration degreesConf = configuration.getChild(ShareableCourseConstants.DEGREES_FIELD_ATTRIBUTE_NAME);
        String degreeValuesAsString = degreesConf.getValue(StringUtils.EMPTY);
        if (__OWN_KEY.equals(degreeValuesAsString))
        {
            return new ShareableField(true, Set.of());
        }
        else
        {
            List<String> valuesAsList = Arrays.asList(StringUtils.split(degreeValuesAsString, ","));
            Set<String> degreeValues = valuesAsList.stream()
                .map(StringUtils::trim)
                .filter(StringUtils::isNotBlank)
                .map(d -> _getItemIdFromCDM(d, OdfReferenceTableHelper.DEGREE))
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
            
            return new ShareableField(false, degreeValues);
        }
    }
    
    /**
     * Parse configuration to get period field 
     * @param configuration the configuration
     * @return the period field
     */
    protected ShareableField _parsePeriodField(Configuration configuration)
    {
        Configuration periodsConf = configuration.getChild(ShareableCourseConstants.PERIODS_FIELD_ATTRIBUTE_NAME);
        String periodValuesAsString = periodsConf.getValue(StringUtils.EMPTY);
        if (__OWN_KEY.equals(periodValuesAsString))
        {
            return new ShareableField(true, Set.of());
        }
        else
        {
            List<String> valuesAsList = Arrays.asList(StringUtils.split(periodValuesAsString, ","));
            Set<String> periodValues = valuesAsList.stream()
                .map(StringUtils::trim)
                .filter(StringUtils::isNotBlank)
                .map(p -> _getItemIdFromCDM(p, OdfReferenceTableHelper.PERIOD))
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
            
            return new ShareableField(false, periodValues);
        }
    }
    
    /**
     * Parse configuration to get orgUnit field 
     * @param configuration the configuration
     * @return the orgUnit field
     */
    protected ShareableField _parseOrgUnitField(Configuration configuration)
    {
        Configuration orgUnitsConf = configuration.getChild(ShareableCourseConstants.ORGUNITS_FIELD_ATTRIBUTE_NAME);
        String orgUnitValuesAsString = orgUnitsConf.getValue(StringUtils.EMPTY);
        if (__OWN_KEY.equals(orgUnitValuesAsString))
        {
            return new ShareableField(true, Set.of());
        }
        else
        {
            List<String> valuesAsList = Arrays.asList(StringUtils.split(orgUnitValuesAsString, ","));
            Set<String> orgUnitValues = valuesAsList.stream()
                    .map(StringUtils::trim)
                    .filter(StringUtils::isNotBlank)
                    .filter(this::_isContentExist)
                    .collect(Collectors.toSet());
            
            return new ShareableField(false, orgUnitValues);
        }
    }
    
    private String _getItemIdFromCDM(String cdmValue, String contentType)
    {
        OdfReferenceTableEntry content = _odfRefTableHelper.getItemFromCDM(contentType, cdmValue);
        if (content == null)
        {
            getLogger().warn("Find a wrong data in the shareable configuration : can't get content from cdmValue {}", cdmValue);
            return null;
        }
        
        return content.getId();
    }
    
    private boolean _isContentExist(String contentId)
    {
        try
        {
            _resolver.resolveById(contentId);
            return true;
        }
        catch (Exception e) 
        {
            getLogger().warn("Find a wrong data in the shareable configuration : can't get content from id {}", contentId);
            return false;
        }
    }
    
    /**
     * Initialize the shareable course fields
     * @param courseContent the course content to initialize
     * @param courseListContent the course list parent. Can be null
     * @param user the user who initialize the shareable fields
     * @param ignoreRights true to ignore user rights
     * @return <code>true</code> if there are changes
     */
    public boolean initializeShareableFields(Course courseContent, CourseList courseListContent, UserIdentity user, boolean ignoreRights)
    {
        List<CourseList> courseListContents = courseListContent != null ? Collections.singletonList(courseListContent) : List.of();
        return initializeShareableFields(courseContent, courseListContents, user, ignoreRights);
    }
    
    /**
     * Initialize the shareable course fields
     * @param courseContent the course content to initialize
     * @param courseListContents the list of course list parents. Can be empty
     * @param user the user who initialize the shareable fields
     * @param ignoreRights true to ignore user rights
     * @return <code>true</code> if there are changes
     */
    public boolean initializeShareableFields(Course courseContent, List<CourseList> courseListContents, UserIdentity user, boolean ignoreRights)
    {
        boolean hasChanges = false;
        if (handleShareableCourse())
        {
            hasChanges = true;
            
            Set<Program> parentPrograms = new HashSet<>();
            Set<Container> parentContainers = new HashSet<>();
            for (CourseList courseList : courseListContents)
            {
                parentPrograms.addAll(_odfHelper.getParentPrograms(courseList));
                parentContainers.addAll(_odfHelper.getParentContainers(courseList));
            }
            
            ShareableField programField = _shareableCourseConfiguration.getProgramField();
            Set<String> programs = programField.ownContext() ? getProgramIds(parentPrograms) : programField.getDefaultValues();
            if (!programs.isEmpty())
            {
                courseContent.setValue(ShareableCourseConstants.PROGRAMS_FIELD_ATTRIBUTE_NAME, programs.toArray(new String[programs.size()]));
            }
            
            ShareableField degreeField = _shareableCourseConfiguration.getDegreeField();
            Set<String> degrees = degreeField.ownContext() ? getDegrees(parentPrograms) : degreeField.getDefaultValues();
            if (!degrees.isEmpty())
            {
                courseContent.setValue(ShareableCourseConstants.DEGREES_FIELD_ATTRIBUTE_NAME, degrees.toArray(new String[degrees.size()]));
            }
            
            ShareableField periodField = _shareableCourseConfiguration.getPeriodField();
            Set<String> periods = periodField.ownContext() ? getPeriods(parentContainers) : periodField.getDefaultValues();
            if (!periods.isEmpty())
            {
                courseContent.setValue(ShareableCourseConstants.PERIODS_FIELD_ATTRIBUTE_NAME, periods.toArray(new String[periods.size()]));
            }
            
            ShareableField orgUnitField = _shareableCourseConfiguration.getOrgUnitField();
            Set<String> orgUnits = orgUnitField.ownContext() ? getOrgUnits(parentPrograms) : orgUnitField.getDefaultValues();
            if (!orgUnits.isEmpty())
            {
                courseContent.setValue(ShareableCourseConstants.ORGUNITS_FIELD_ATTRIBUTE_NAME, orgUnits.toArray(new String[orgUnits.size()]));
            }
            
            if (_shareableCourseConfiguration.autoValidated())
            {
                _shareableCourseStatus.setWorkflowStateAttribute(courseContent, LocalDate.now(), user, ShareableStatus.VALIDATED, null, ignoreRights);
            }
        }
        
        return hasChanges;
    }
    
    /**
     * True if shareable course fields match with the course list
     * @param courseContent the shareable course
     * @param courseList the course list
     * @return <code>true</code> if shareable course fields match with the course list
     */
    public boolean isShareableFieldsMatch(Course courseContent, CourseList courseList)
    {
        if (_shareableCourseStatus.getShareableStatus(courseContent) != ShareableStatus.VALIDATED)
        {
            return false;
        }

        List<String> programValues = ContentDataHelper.getContentIdsListFromMultipleContentData(courseContent, ShareableCourseConstants.PROGRAMS_FIELD_ATTRIBUTE_NAME);
        List<String> degreeValues = ContentDataHelper.getContentIdsListFromMultipleContentData(courseContent, ShareableCourseConstants.DEGREES_FIELD_ATTRIBUTE_NAME);
        List<String> orgUnitValues = ContentDataHelper.getContentIdsListFromMultipleContentData(courseContent, ShareableCourseConstants.ORGUNITS_FIELD_ATTRIBUTE_NAME);
        List<String> periodValues = ContentDataHelper.getContentIdsListFromMultipleContentData(courseContent, ShareableCourseConstants.PERIODS_FIELD_ATTRIBUTE_NAME);

        if (programValues.isEmpty() && degreeValues.isEmpty() && orgUnitValues.isEmpty() && periodValues.isEmpty())
        {
            return true;
        }
        
        boolean isSharedWith = true;
        if (!periodValues.isEmpty())
        {
            Set<Container> parentContainers = _odfHelper.getParentContainers(courseList);
            Set<String> courseListPeriods = getPeriods(parentContainers);
            
            isSharedWith = CollectionUtils.containsAny(courseListPeriods, periodValues) && isSharedWith;
        }
        
        if (isSharedWith)
        {
            Set<Program> parentPrograms = _odfHelper.getParentPrograms(courseList);

            if (!programValues.isEmpty())
            {
                Set<String> courseListprogramIds = getProgramIds(parentPrograms);
                isSharedWith = CollectionUtils.containsAny(courseListprogramIds, programValues);
            }
            
            if (!degreeValues.isEmpty())
            {
                Set<String> courseListDegrees = getDegrees(parentPrograms);
                isSharedWith = isSharedWith && CollectionUtils.containsAny(courseListDegrees, degreeValues);
            }
            
            if (!orgUnitValues.isEmpty())
            {
                Set<String> courseListorgUnitIds = getOrgUnits(parentPrograms);
                isSharedWith = isSharedWith && CollectionUtils.containsAny(courseListorgUnitIds, orgUnitValues);
            }
        }
        
        return isSharedWith;
    }
    
    /**
     * Get programs ids from programs
     * @param programs the list of programs
     * @return the list of progam ids
     */
    public Set<String> getProgramIds(Set<Program> programs)
    {
        return programs.stream()
                .map(Program::getId)
                .collect(Collectors.toSet());
    }
    
    /** 
     * Get degrees from programs
     * @param programs the list of programs
     * @return the list of degrees
     */
    public Set<String> getDegrees(Set<Program> programs)
    {
        return programs.stream()
                .map(Program::getDegree)
                .filter(StringUtils::isNotBlank)
                .collect(Collectors.toSet());
    }
    
    /** 
     * Get periods from containers
     * @param containers the list of containers
     * @return the list of periods
     */
    public Set<String> getPeriods(Set<Container> containers)
    {
        return containers.stream()
                .map(Container::getPeriod)
                .filter(StringUtils::isNotBlank)
                .collect(Collectors.toSet());
    }
    
    /** 
     * Get orgUnits from programs
     * @param programs the list of programs
     * @return the list of orgUnits
     */
    public Set<String> getOrgUnits(Set<Program> programs)
    {
        return programs.stream()
                .map(Program::getOrgUnits)
                .flatMap(Collection::stream)
                .filter(StringUtils::isNotBlank)
                .collect(Collectors.toSet());
    }
    
    /**
     * Return true if shareable course are handled
     * @return <code>true</code> if shareable course are handled
     */
    public boolean handleShareableCourse()
    {
        boolean handleShareableCourse = Config.getInstance().getValue("odf.shareable.course.enable");
        return handleShareableCourse;
    }
    
    /**
     * Get the {@link Course}s with shareable filters matching the given arguments
     * @param programId the program id. Can be null
     * @param degreeId the degree id. Can be null
     * @param periodId the period id. Can be null
     * @param orgUnitId the orgUnit id. Can be null
     * @return The matching courses
     */
    public AmetysObjectIterable<Course> getShareableCourses(String programId, String degreeId, String periodId, String orgUnitId)
    {
        List<Expression> exprs = new ArrayList<>();
        exprs.add(new ContentTypeExpression(Operator.EQ, CourseFactory.COURSE_CONTENT_TYPE));
        
        if (StringUtils.isNotEmpty(programId))
        {
            exprs.add(new StringExpression(ShareableCourseConstants.PROGRAMS_FIELD_ATTRIBUTE_NAME, Operator.EQ, programId));
        }
        
        if (StringUtils.isNotEmpty(degreeId))
        {
            exprs.add(new StringExpression(ShareableCourseConstants.DEGREES_FIELD_ATTRIBUTE_NAME, Operator.EQ, degreeId));
        }
        
        if (StringUtils.isNotEmpty(periodId))
        {
            exprs.add(new StringExpression(ShareableCourseConstants.PERIODS_FIELD_ATTRIBUTE_NAME, Operator.EQ, periodId));
        }
        
        if (StringUtils.isNotEmpty(orgUnitId))
        {
            exprs.add(new StringExpression(ShareableCourseConstants.ORGUNITS_FIELD_ATTRIBUTE_NAME, Operator.EQ, orgUnitId));
        }
        
        Expression expr = new AndExpression(exprs.toArray(new Expression[exprs.size()]));
        
        String xpathQuery = ContentQueryHelper.getContentXPathQuery(expr);
        return _resolver.query(xpathQuery);
    } 
    
    
    private static class ShareableField
    {
        private boolean _own;
        private Set<String> _values;
        
        public ShareableField(boolean own, Set<String> values)
        {
            _own = own;
            _values = values;
        }
        
        public boolean ownContext()
        {
            return _own;
        }
        
        public Set<String> getDefaultValues()
        {
            return _values;
        }
    }
    
    private static class ShareableConfiguration
    {
        private ShareableField _programField;
        private ShareableField _degreeField;
        private ShareableField _orgUnitField;
        private ShareableField _periodField;
        private boolean _autoValidated;
        
        public ShareableConfiguration(ShareableField programField, ShareableField degreeField, ShareableField orgUnitField, ShareableField periodField, boolean autoValidated)
        {
            _programField = programField;
            _degreeField = degreeField;
            _orgUnitField = orgUnitField;
            _periodField = periodField;
            _autoValidated = autoValidated;
        }
        
        public boolean autoValidated()
        {
            return _autoValidated;
        }
        
        public ShareableField getProgramField()
        {
            return _programField;
        }
        
        public ShareableField getDegreeField()
        {
            return _degreeField;
        }
        
        public ShareableField getOrgUnitField()
        {
            return _orgUnitField;
        }
        
        public ShareableField getPeriodField()
        {
            return _periodField;
        }
    }
    
}
