/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.odf.source;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;

import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Default implementation of {@link ODFViewSelector}.
 * First looks the wanted file in context://WEB-INF/stylesheets/path_to_file
 * If it's not present, it will look in the ODF plugin.
 */
public class DefaultODFView extends AbstractLogEnabled implements ODFViewSelector, Serviceable
{
    
    /** The source resolver */
    protected SourceResolver _resolver;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
    }
    
    @Override
    public Source getSource(String pluginName, String filePath) throws IOException
    {
        for (String location : getLocations(pluginName, filePath))
        {
            try
            {
                Source source = _resolver.resolveURI(location);
                if (!source.exists())
                {
                    getLogger().debug("Failed to find a stylesheet at '{}'.", location);
                }
                else
                {
                    getLogger().debug("Using source located at '{}'.", location);
                    return source;
                }
            }
            catch (IOException e)
            {
                getLogger().debug("Resolving protocol failed for resolving '{}'.", location);
            }
        }
        
        // Should never occur because of the default stylesheet.
        throw new IOException("Can't find a stylesheet for: " + filePath);
    }
    
    /**
     * Returns the ordered list of URI to be tested to find the corresponding view.
     * @param pluginName the plugin name
     * @param filePath the requested location.
     * @return a list of possible URIs for the resource.
     */
    protected List<String> getLocations(String pluginName, String filePath)
    {
        ArrayList<String> locations = new ArrayList<>();
        
        // First search in the WEB-INF folder.
        locations.add("context://WEB-INF/" + filePath);
        
        // Then look in the current plugin if not null
        if (pluginName != null)
        {
            locations.add("plugin:" + pluginName + "://" + filePath);
        }
        
        locations.addAll(_getDefaultSourceURIs(pluginName, filePath));
        
        return locations;
    }
    
    /**
     * Get the default URIs
     * @param pluginName The current plugin name
     * @param filePath the requested location.
     * @return a list of URIs
     */
    protected List<String> _getDefaultSourceURIs(String pluginName, String filePath)
    {
        List<String> locations = new ArrayList<>();
        
        // For legacy, look in CMS plugin
        if (!"odf".equals(pluginName))
        {
            locations.add("plugin:odf://" + filePath);
        }
        
        return locations;
    }
}
