/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.workflow;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.AbstractContentFunction;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.odf.course.Course;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.coursepart.CoursePartFactory;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.WorkflowException;

/**
 * Do action on CoursePart of given course.
 */
public abstract class AbstractCoursePartFunction extends AbstractContentFunction
{
    /** Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** Content workflow helper */
    protected ContentWorkflowHelper _contentWorkflowHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) smanager.lookup(ContentWorkflowHelper.ROLE);
    }
    
    @Override
    public void execute(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        // Retrieve current content
        WorkflowAwareContent content = getContent(transientVars);
        
        if (content instanceof Course)
        {
            // Build the expression to retrieve course parts which the course is holding
            Expression expression = new AndExpression(
                    new ContentTypeExpression(Operator.EQ, CoursePartFactory.COURSE_PART_CONTENT_TYPE),
                    new StringExpression(CoursePart.COURSE_HOLDER, Operator.EQ, content.getId())
            );
            String xpathQuery = ContentQueryHelper.getContentXPathQuery(expression);
            
            int actionId = _getCoursePartActionId();
            AmetysObjectIterable<CoursePart> courseParts = _resolver.query(xpathQuery);
            for (CoursePart coursePart : courseParts)
            {
                if (_contentWorkflowHelper.isAvailableAction(coursePart, actionId))
                {
                    _contentWorkflowHelper.doAction(coursePart, actionId);
                }
            }
        }
    }
    
    @Override
    public FunctionType getFunctionExecType()
    {
        return FunctionType.POST;
    }
    
    /**
     * Get the id of the action to do for course part
     * @return the id of the action to do for course part
     */
    protected abstract Integer _getCoursePartActionId();
}
