/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This tool is a tool that display a tree of the current selected program.
 */
Ext.define('Ametys.plugins.odf.tree.AbstractODFTreeTool', {
	extend: "Ametys.plugins.contentstree.ContentsTreeTool",
    
    /**
     * @property {Object[]} _indicators See {@link #cfg-indicators}.
     * @private
     */
    
    /**
     * @cfg {Boolean} [enableValidationCheck=false] Set to 'true' to enable the global validation state on the root content
     */
    /**
     * @property {Boolean} _enableValidationCheck See {@link #cfg-enableValidationCheck}.
     * @private
     */
    /**
     * @cfg {Boolean} [checkValidationAtStartup=false] Set to 'true' to start the global validation check at startup. Use only if {@link #cfg-enableValidationCheck} is true.
     */
    /**
     * @property {Boolean} _checkValidationAtStartup See {@link #cfg-checkValidationAtStartup}.
     * @private
     */
    
    constructor: function(config)
    {
        this._enableValidationCheck = config.enableValidationCheck == true;
        this._checkValidationAtStartup = this._enableValidationCheck && config.checkValidationAtStartup == true;
        
        this.callParent(arguments);
        
        this._indicators = this._getIndicators(config);
        
        Ametys.message.MessageBus.on(Ametys.message.Message.WORKFLOW_CHANGED, this._onWorkflowChanged, this);
    },
    
    _getIndicators: function(config)
    {
        return config.indicators ? JSON.parse(config.indicators) : [];
    },
    
    /**
     * @protected
     * Create the tree panel
     * @return {Ext.tree.Panel} the tree panel
     */
    _createTree: function()
    {
        return Ext.create('Ametys.plugins.odf.tree.ODFContentsTreePanel', this._getTreeConfig());
    },
    
    _getTreeConfig: function()
    {
        var treeCfg = this.callParent(arguments);
        return Ext.apply(treeCfg, {
            stateful: true,
            stateId: this.getId() + '$odf-tree',
            indicators: this._indicators,
            enableValidationCheck: this._enableValidationCheck
        });
    },
    
    /**
     * Listener when something has been modified on the workflow
     * @param {Ametys.message.Message} [message] The bus message. Can be null to get the last selection message
     * @private
     */
    _onWorkflowChanged: function(message)
    {
        this._onModified(message);
    },
    
    /**
     * Set the content root node
     * @param {String} contentId The id of root content
     * @param {String} pathToSelect The path to select
     */
    _setContentRootNode: function(contentId, pathToSelect)
    {
        if (this._enableValidationCheck)
        {
            // reset the precede validation state
            this._treePanel.resetValidationState();
        }
        
        this._treePanel.setContentRootNode(
            contentId, 
            this._getRootNodeInfo(),
            Ext.bind(this._setContentRootNodeCb, this, [pathToSelect])
        );
    },
    
    _setContentRootNodeCb: function(pathToSelect)
    {
        this._refreshed(pathToSelect);
        
        if (this._checkValidationAtStartup)
        {
            this._treePanel.checkValidationState();
        }
    },
    
    /**
     * @template
     * Get the list of content types to open on double-click
     * @return {String[]} The array of openable content types
     */
    getOpenableContentTypes: function()
    {
        return [];
    },
    
	_onDoubleClick: function(tree, record, item, index, e, eOpts)
    {
    	var contentId = record.get('contentId');
    	var types = record.get('contenttypesIds') || [];
        var openableTypes = this.getOpenableContentTypes();
        
    	if (contentId && Ext.Array.intersect(openableTypes, types).length > 0)
    	{
    		Ametys.tool.ToolsManager.openTool('uitool-content', {id: contentId, mode: 'view'});
    	}
    }
});