/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.program;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.jcr.Node;

import org.apache.commons.lang.StringUtils;

import org.ametys.cms.data.ContentDataHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableDefaultContent;
import org.ametys.odf.content.code.DisplayCodeProperty;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.runtime.model.exception.UndefinedItemPathException;

/**
 * Common implementation of a {@link Content} which is part of a program
 * @param <F> the actual type of factory.
 */
public abstract class AbstractProgramPart<F extends ProgramPartFactory> extends ModifiableDefaultContent<F> implements ProgramPart
{
    /**
     * Constructor
     * @param node The JCR node
     * @param parentPath The parent path
     * @param factory The factory
     */
    public AbstractProgramPart(Node node, String parentPath, F factory)
    {
        super(node, parentPath, factory);
    }
    
    @Override
    public List<ProgramPart> getProgramPartParents()
    {
        return Arrays.stream(getValue(PARENT_PROGRAM_PARTS, false, new ContentValue[0]))
                .map(ContentValue::getContentIfExists)
                .flatMap(Optional::stream)
                .map(ProgramPart.class::cast)
                .collect(Collectors.toList());
    }
    
    @Override
    public String getCatalog()
    {
        return getValue(CATALOG);
    }

    @Override
    public void setCatalog(String catalog) throws AmetysRepositoryException
    {
        setValue(CATALOG, catalog);
    }
    
    @Override
    public String getCode()
    {
        return getValue(CODE, false, StringUtils.EMPTY);
    }
    
    @Override
    public void setCode(String code) throws AmetysRepositoryException
    {
        setValue(CODE, code);
    }
    
    public String getDisplayCode()
    {
        return getValue(DisplayCodeProperty.PROPERTY_NAME, false, StringUtils.EMPTY);
    }
    
    public boolean isPublishable()
    {
        return getInternalDataHolder().getValue(PUBLISHABLE, true);
    }
    
    public void setPublishable(boolean isPublishable)
    {
        getInternalDataHolder().setValue(PUBLISHABLE, isPublishable);
    }
    
    public List<String> getOrgUnits()
    {
        try
        {
            return ContentDataHelper.getContentIdsListFromMultipleContentData(this, ORG_UNITS_REFERENCES);
        }
        catch (UndefinedItemPathException e)
        {
            return Collections.EMPTY_LIST; // this attribute is not part of model
        }
    }
    
    // --------------------------------------------------------------------------------------//
    // CDM-fr
    // --------------------------------------------------------------------------------------//
    @Override
    public String getCDMId()
    {
        String cdmCode = getCdmCode();
        if (StringUtils.isEmpty(cdmCode))
        {
            return "FRUAI" + _getFactory()._getRootOrgUnitRNE() + getCDMType() + getCode();
        }
        return cdmCode;
    }
    
    /**
     * Get the type of this {@link ProgramPart} in CDM. This type compose the CDM identifier
     * @return the type
     */
    protected abstract String getCDMType();
    
    @Override
    public String getCdmCode()
    {
        return getValue(CDM_CODE, false, StringUtils.EMPTY);
    }
    
    @Override
    public void setCdmCode(String cdmCode)
    {
        setValue(CDM_CODE, cdmCode);
    }
}
