/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.skill.imports.csv;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.program.Program;
import org.ametys.odf.skill.workflow.SkillEditionFunction;
import org.ametys.plugins.contentio.csv.CSVImporter;
import org.ametys.plugins.contentio.csv.SynchronizeModeEnumerator.ImportMode;
import org.ametys.plugins.repository.query.expression.BooleanExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.ViewItem;

/**
 * Import skills from an uploaded CSV file.
 */
public class SkillCSVImporter extends CSVImporter
{
    
    /** Avalon Role */
    @SuppressWarnings("hiding")
    public static final String ROLE = SkillCSVImporter.class.getName();
    /** The content types helper */
    protected ContentTypesHelper _contentTypesHelper;
    /** The catalog manager */
    protected CatalogsManager _catalogsManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _contentTypesHelper = (ContentTypesHelper) smanager.lookup(ContentTypesHelper.ROLE);
        _catalogsManager = (CatalogsManager) smanager.lookup(CatalogsManager.ROLE);
    }

    @Override
    protected List<Expression> additionalExpressionToRetrieveContent(Map<String, Object> mapping, Map<String, String> row, ContentType contentType, Optional<? extends Content> parentContent, Map<String, Object> additionalTransientVars, List<String> valuesUsedAsIds)
    {
        List<Expression> expression = new ArrayList<>();
        
        // If we are importing a macro skill, we need to set transversal or the parent program
        if (SkillEditionFunction.MACRO_SKILL_TYPE.equals(contentType.getId()))
        {
            if (parentContent.isPresent() && parentContent.get() instanceof Program programParent)
            {
                expression.add(new StringExpression("parentProgram", Operator.EQ, programParent.getId()));
                valuesUsedAsIds.add("parentProgram");
            }
            else
            {
                // If no parent is set, then we are trying to import a transversal skill
                expression.add(new BooleanExpression("transversal", Operator.EQ, true));
                valuesUsedAsIds.add("transversal");
            }
        }
        // If we are importing a micro skill, we need to set the parent macro skill
        else if (SkillEditionFunction.MICRO_SKILL_TYPE.equals(contentType.getId()) && parentContent.isPresent() && _contentTypesHelper.isInstanceOf(parentContent.get(), SkillEditionFunction.MACRO_SKILL_TYPE))
        {
            Content macroParent = parentContent.get();
            expression.add(new StringExpression("parentMacroSkill", Operator.EQ, macroParent.getId()));
            valuesUsedAsIds.add("parentMacroSkill");
        }
            
        return expression;
    }
    
    @Override
    protected SynchronizeResult processContent(View view, Map<String, String> row, ContentType contentType, Map<String, Object> mapping, int createAction, int editAction, String workflowName, String language, List<ViewItem> errors, ImportMode importMode, Optional< ? extends Content> parentContent, Map<String, Object> additionalTransientVars) throws Exception
    {
        // If we are trying to import a transversal skill, check that the catalog is set to an existing one
        if (parentContent.isEmpty())
        {
            // Check that the catalog exists
            String catalogName = row.get(ImportSkillsCSVFileHelper.SKILL_CATALOG_ATTRIBUTE_PATH);
            if (_catalogsManager.getCatalog(catalogName) == null)
            {
                throw new IllegalArgumentException("The catalog '" + catalogName + "' does not exist.");
            }
        }
        
        return super.processContent(view, row, contentType, mapping, createAction, editAction, workflowName, language, errors, importMode, parentContent, additionalTransientVars);
    }
}
