/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Singleton class to remove or add ODF content from publication
 * @private
 */
Ext.define('Ametys.plugins.odf.content.PublicationAction', {
    singleton: true,
    
    /**
	 * Action function to be called by the controller.
	 * Will add or remove from the publication the contents registered by the controller.
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller the controller calling this function
     * @param {Boolean} state the press-state of the controller
	 */
	act: function (controller, state)
	{
        var isPublishable = !state;
        var contentIds = isPublishable ? controller.getNotPublishableContentIds() : controller.getPublishableContentIds();
        
        if (!isPublishable)
        {
            var me = this;
            Ametys.Msg.confirm("{{i18n PLUGINS_ODF_PUBLISHABLE_STATE_NOT_PUBLISHABLE_LABEL}}",
                "{{i18n PLUGINS_ODF_PUBLISHABLE_STATE_NOT_PUBLISHABLE_GLOBAL_CONFIRM}}",
                function(answer)
                {
                    if (answer == 'yes')
                    {
                        Ext.create("Ametys.message.Message", {
                            type: Ametys.message.Message.WORKFLOW_CHANGING,
                            
                            targets: {
                                id: Ametys.message.MessageTarget.CONTENT,
                                parameters: { ids: contentIds }
                            }
                        });
        
                        me._setPublishableState(contentIds, isPublishable);
                    }
                }
            );
        }
        else
        {
            this._setPublishableState(contentIds, isPublishable);
        }
	},
	
	/**
	 * @private
     * Will add or remove from the publication the contents registered by the controller.
     * @param {String[]} contentIds the content ids
     * @param {Boolean} isPublishable true if the content are publishable
     */
	_setPublishableState: function (contentIds, isPublishable)
	{
        Ametys.data.ServerComm.callMethod({
            role: "org.ametys.odf.workflow.ODFWorkflowHelper",
            methodName: "setPublishableState",
            priority: Ametys.data.ServerComm.PRIORITY_LONG_REQUEST,
            parameters: [contentIds, isPublishable],
            callback: {
                scope: this,
                handler: this._setPublishableStateCB,
                arguments: [contentIds, isPublishable]
            },
            waitMessage: isPublishable,
            errorMessage: false
        });
    },
	
	/**
	 * @private
     * Callback function after setting publishable state
     * @param {Object} response The server response
     * @param {Object} args The callback arguments
     */
    _setPublishableStateCB: function (response, args)
    {
        var me = this;
        var contentIds = args[0];
        var isPublishable = args[1];
        Ametys.cms.content.ContentDAO.getContents(contentIds, function(contents) {
            
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.WORKFLOW_CHANGED,
                
                targets: {
                    id: Ametys.message.MessageTarget.CONTENT,
                    parameters: { contents: contents }
                }
            });
            
            Ext.Array.forEach(contents, function(content) {
            
                var result = response[content.getId()];
                var unpublishedContents = result['unpublishedContents'];
                var contentsInError = result['contentsInError'];
                var contentsWithRightError = result['contentsWithRightError'];
                
                var titleMsg = isPublishable ? "{{i18n PLUGINS_ODF_PUBLISHABLE_STATE_PUBLISHABLE_LABEL}}" : "{{i18n PLUGINS_ODF_PUBLISHABLE_STATE_NOT_PUBLISHABLE_LABEL}}";
                if (contentsInError.length == 0 && contentsWithRightError.length == 0)
                {
                    var msg = isPublishable ? me._getPublishSuccessMsg(content) : me._getUnpublishSuccessMsg(content, unpublishedContents);
                    Ametys.notify({
                        type: 'info',
                        iconGlyph: 'ametysicon-sign-raw-check',
                        title: titleMsg,
                        description: msg
                    });
                }
                else
                {
                    var msg = isPublishable ? me._getPublishErrorMsg(contentsInError) : me._getUnpublishErrorMsg(content, contentsInError, contentsWithRightError);
                    Ametys.notify({
                        type: isPublishable ? 'error' : 'warn',
                        iconGlyph: 'ametysicon-sign-raw-check',
                        title: titleMsg,
                        description: msg
                    });
                }
                
            });
        });
    },
    
    /**
     * @private
     * Get unpublish success message
     * @param {Content} content The content
     * @param {String[]} unpublishedContents The unpublished content ids
     */
    _getUnpublishSuccessMsg: function(content, unpublishedContents)
    {
        var infoMsg = Ext.String.format("{{i18n PLUGINS_ODF_PUBLISHABLE_STATE_NOT_PUBLISHABLE_SUCCESS}}", content.getTitle());
        if (unpublishedContents.length > 1)
        {
            infoMsg += Ext.String.format("{{i18n PLUGINS_ODF_PUBLISHABLE_STATE_NOT_PUBLISHABLE_SEVERAL_CONTENTS}}", unpublishedContents.length);
        }
        else if (unpublishedContents.length == 1)
        {
            infoMsg += "{{i18n PLUGINS_ODF_PUBLISHABLE_STATE_NOT_PUBLISHABLE_ONE_CONTENT}}";
        }
        return infoMsg;
    },
    
    /**
     * @private
     * Get unpublish error message
     * @param {Content} content The content
     * @param {Object[]} contentsInError The contents is error
     */
    _getUnpublishErrorMsg: function(content, contentsInError, contentsWithRightError)
    {
        var warnMsg = Ext.String.format("{{i18n PLUGINS_ODF_PUBLISHABLE_STATE_NOT_PUBLISHABLE_WARN}}", content.getTitle());
        if (contentsInError.length != 0)
        {
            warnMsg += "<br/><br/>"
            warnMsg += "{{i18n PLUGINS_ODF_PUBLISHABLE_STATE_NOT_PUBLISHABLE_WARN_START}}";
            warnMsg += "<ul>";
        
            Ext.Array.each(contentsInError, function(contentInError) {
                var href = "javascript:(function(){parent.Ametys.tool.ToolsManager.openTool('uitool-content', {id:'" + contentInError.id + "'});})()";
                warnMsg += '<li><a href="' + href + '">' + contentInError.title + (contentInError.code ? ' (' + contentInError.code + ')' : '') + '</a></li>';
            });
            warnMsg += "</ul>"
            warnMsg += "{{i18n PLUGINS_ODF_PUBLISHABLE_STATE_NOT_PUBLISHABLE_WARN_END}}";
        }
        
        if (contentsWithRightError.length != 0)
        {
            warnMsg += "<br/><br/>"
            warnMsg += contentsInError.length != 0 ? "{{i18n PLUGINS_ODF_PUBLISHABLE_STATE_NOT_PUBLISHABLE_RIGHT_WARN_START_1}}" : "{{i18n PLUGINS_ODF_PUBLISHABLE_STATE_NOT_PUBLISHABLE_RIGHT_WARN_START_2}}";
            warnMsg += "<ul>";
        
            Ext.Array.each(contentsWithRightError, function(contentInError) {
                var href = "javascript:(function(){parent.Ametys.tool.ToolsManager.openTool('uitool-content', {id:'" + contentInError.id + "'});})()";
                warnMsg += '<li><a href="' + href + '">' + contentInError.title + (contentInError.code ? ' (' + contentInError.code + ')' : '') + '</a></li>';
            });
            warnMsg += "</ul>"
        }
        
        return warnMsg;
    },
    
    /**
     * @private
     * Get publish success message
     * @param {Content} content The content
     */
    _getPublishSuccessMsg: function(content)
    {
        return Ext.String.format("{{i18n PLUGINS_ODF_PUBLISHABLE_STATE_PUBLISHABLE_SUCCESS}}", content.getTitle());
    },
    
    /**
     * @private
     * Get publish success message
     * @param {Object[]} contentsInError The contents is error
     */
    _getPublishErrorMsg: function(contentsInError)
    {
        var href = "javascript:(function(){parent.Ametys.tool.ToolsManager.openTool('uitool-content', {id:'" + contentsInError[0].id + "'});})()";
        var title = '<a href="' + href + '">' + contentsInError[0].title + (contentsInError[0].code ? ' (' + contentsInError[0].code + ')' : '') + '</a>';
        return Ext.String.format("{{i18n PLUGINS_ODF_PUBLISHABLE_STATE_PUBLISHABLE_ERROR}}", title);
    },
});