/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.query.InvalidQueryException;
import javax.jcr.query.Query;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.cms.data.holder.group.IndexableRepeaterEntry;
import org.ametys.cms.data.holder.group.ModifiableIndexableRepeater;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.cms.workflow.EditContentFunction;
import org.ametys.cms.workflow.ValidateContentFunction;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.data.type.EducationalPathRepositoryElementType;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.holder.impl.DataHolderHelper;
import org.ametys.plugins.repository.data.holder.values.SynchronizableValue;
import org.ametys.plugins.repository.data.holder.values.SynchronizationResult;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.plugins.repository.provider.AbstractRepository;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.plugins.workflow.component.CheckRightsCondition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.WorkflowException;

/**
 * Helper for manipulating {@link EducationalPath}
 */
public class EducationalPathHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** The avalon role */
    public static final String ROLE = EducationalPathHelper.class.getName();
    
    /** Constant to get/set the ancestor path (may be partial) of a program item in request attribute */
    public static final String PROGRAM_ITEM_ANCESTOR_PATH_REQUEST_ATTR = EducationalPathHelper.class.getName() + "$ancestorPath";
    
    /** Constant to get/set the root program item in request attribute */
    public static final String ROOT_PROGRAM_ITEM_REQUEST_ATTR = EducationalPathHelper.class.getName() + "$rootProgramItem";
    
    private AmetysObjectResolver _resolver;
    private Repository _repository;

    private ContentWorkflowHelper _workflowHelper;

    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _repository = (Repository) manager.lookup(AbstractRepository.ROLE);
        _workflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
    }
    
    /**
     * Determines if program items is part of a same educational path value
     * @param programItems the program items to search into a educational path value
     * @return true if the program items is part of a educational path value
     * @throws RepositoryException if failed to get educational path nodes
     */
    public boolean isPartOfEducationalPath(ProgramItem... programItems) throws RepositoryException
    {
        String[] programItemIds = Stream.of(programItems)
                                        .map(ProgramItem::getId)
                                        .toArray(String[]::new);
        return isPartOfEducationalPath(programItemIds);
    }
    
    /**
     * Determines if program items is part of a same educational path  value
     * @param programItemIds the id of program items to search into a educational path  value
     * @return true if the program items is part of a educational path  value
     * @throws RepositoryException if failed to get educational path nodes
     */
    public boolean isPartOfEducationalPath(String... programItemIds) throws RepositoryException
    {
        NodeIterator nodeIterator = _getEducationalPathNodes(programItemIds);
        return nodeIterator.hasNext();
    }
    
    /**
     * Get the educational paths that reference the given program items
     * @param programItems the referenced program items to search into a educational path
     * @return a Map with program items referencing the given program items in a educational path attribute as key and the list of educational path references as value
     * @throws RepositoryException if an error occurred while retrieving educational path references
     */
    public Map<ProgramItem, List<EducationalPathReference>> getEducationalPathReferences(ProgramItem... programItems) throws RepositoryException
    {
        String[] programItemIds = Stream.of(programItems)
                .map(ProgramItem::getId)
                .toArray(String[]::new);
        return getEducationalPathReferences(programItemIds);
    }
    
    /**
     * Get the educational paths that reference the given program items
     * @param programItemIds the id of program items to search into a educational path
     * @return a Map with program items referencing the given program items in a educational path attribute as key and the list of educational path references as value
     * @throws RepositoryException if an error occurred while retrieving educational path references
     */
    public Map<ProgramItem, List<EducationalPathReference>> getEducationalPathReferences(String... programItemIds) throws RepositoryException
    {
        Map<ProgramItem, List<EducationalPathReference>> references = new HashMap<>();
        
        NodeIterator educationalPathNodesIterator = _getEducationalPathNodes(programItemIds);
        
        while (educationalPathNodesIterator.hasNext())
        {
            Node educationalPathNode = educationalPathNodesIterator.nextNode();
            String educationalPathAttributeName = StringUtils.substringAfter(educationalPathNode.getName(), RepositoryConstants.NAMESPACE_PREFIX + ":");
            
            Node repeaterEntryNode = educationalPathNode.getParent();
            
            Pair<ProgramItem, String> resolvedEntryPath = _resolveEntryPath(repeaterEntryNode);
            
            ProgramItem refProgramItem = resolvedEntryPath.getLeft();
            if (!references.containsKey(refProgramItem))
            {
                references.put(refProgramItem, new ArrayList<>());
            }
            
            EducationalPath value = ((Content) refProgramItem).getValue(resolvedEntryPath.getRight() + "/" + educationalPathAttributeName);
            references.get(refProgramItem).add(new EducationalPathReference(refProgramItem, resolvedEntryPath.getRight(), educationalPathAttributeName, value));
        }
        
        return references;
    }
    
    /**
     * Remove all repeater entries with a educational path that reference the given program items
     * @param programItems the referenced program items to search into a educational path
     * @throws RepositoryException if an error occurred while retrieving educational path references
     * @throws WorkflowException if failed to do workflow action on modified contents
     */
    public void removeEducationalPathReferences(ProgramItem... programItems) throws RepositoryException, WorkflowException
    {
        String[] programItemIds = Stream.of(programItems)
                .map(ProgramItem::getId)
                .toArray(String[]::new);
        removeEducationalPathReferences(2, programItemIds);
    }
    
    /**
     * Remove all repeater entries with a educational path that reference the given program items
     * @param programItemIds the id program items to search into a educational path
     * @throws RepositoryException if an error occurred while retrieving educational path references
     * @throws WorkflowException if failed to do workflow action on modified contents
     */
    public void removeEducationalPathReferences(List<String> programItemIds) throws RepositoryException, WorkflowException
    {
        removeEducationalPathReferences(2, programItemIds.toArray(String[]::new));
    }
    
    /**
     * Remove all repeater entries with a educational path that reference the given program items
     * @param workflowActionId The id of the workflow action to do
     * @param programItemIds the ids of program items to search into a educational path
     * @throws RepositoryException if an error occurred while retrieving educational path references
     * @throws WorkflowException if failed to do workflow action on modified contents
     */
    public void removeEducationalPathReferences(int workflowActionId, String... programItemIds) throws RepositoryException, WorkflowException
    {
        Map<ProgramItem, List<EducationalPathReference>> educationalPathReferences = getEducationalPathReferences(programItemIds);
        
        for (Entry<ProgramItem, List<EducationalPathReference>> entry : educationalPathReferences.entrySet())
        {
            ProgramItem refProframItem = entry.getKey();
            
            List<EducationalPathReference> references = entry.getValue();
            for (EducationalPathReference educationalPathReference : references)
            {
                String attributeName = educationalPathReference.attributeName();
                EducationalPath value = educationalPathReference.value();
                String repeaterEntryPath = educationalPathReference.repeaterEntryPath(); // (ex: rep1[2]/comp/rep1[1]/rep3[2])
                String repeaterPath = StringUtils.substringBeforeLast(repeaterEntryPath, "["); // rep1[2]/comp/rep1[1]/rep3
                
                // Do NOT use repeater entry path to remove it as entry position can change during removing of entries
                // Find entry with same educational path value
                ModifiableIndexableRepeater repeater = ((ModifiableContent) refProframItem).getRepeater(repeaterPath);
                Optional<Integer> entryPos = repeater.getEntries()
                    .stream()
                    .filter(e -> value.equals(e.getValue(attributeName)))
                    .map(e -> e.getPosition())
                    .findFirst();
                            
                if (entryPos.isPresent())
                {
                    String realEntryPath = repeaterPath + "[" + entryPos.get() + "]";
                    ((ModifiableContent) refProframItem).removeValue(realEntryPath);
                }
            }
            
            _triggerWorkflowAction((Content) refProframItem, 2);
        }
    }
    
    
    /**
     * Remove all repeaters with a educational path in their model
     * @param programItem the program item
     * @throws WorkflowException if failed to do workflow action on modified contents
     */
    public void removeAllRepeatersWithEducationalPath(ProgramItem programItem) throws WorkflowException
    {
        ModifiableContent content = (ModifiableContent) programItem;
        
        boolean needSave = false;
        
        Map<String, Object> educationPaths = DataHolderHelper.findItemsByType(content, EducationalPathRepositoryElementType.EDUCATIONAL_PATH_ELEMENT_TYPE_ID);
        
        // Get path of repeaters with a education path value
        Set<String> repeaterDataPaths = educationPaths.entrySet().stream()
            .filter(e -> e.getValue() instanceof EducationalPath)
            .map(Entry::getKey) // data path
            .map(dataPath -> StringUtils.substring(dataPath, 0, dataPath.lastIndexOf("/"))) // parent data path
            .filter(DataHolderHelper::isRepeaterEntryPath)
            .map(DataHolderHelper::getRepeaterNameAndEntryPosition) // get repeater path
            .map(Pair::getLeft)
            .collect(Collectors.toSet());
        
        for (String repeaterDataPath : repeaterDataPaths)
        {
            RepeaterDefinition repeaterDef = (RepeaterDefinition) content.getDefinition(repeaterDataPath);
            if (content.hasValue(repeaterDataPath) && !_isInRepeaterWithCommonAttribute(repeaterDef)) // do not remove repeaters with common model item !
            {
                content.removeValue(repeaterDataPath);
                needSave = true;
            }
        }
        
        if (needSave)
        {
            _triggerWorkflowAction(content, 2);
        }
    }
    
    private boolean _isInRepeaterWithCommonAttribute(ModelItem modelItem)
    {
        ModelItem parentItem = modelItem;
        while (parentItem != null)
        {
            if (parentItem instanceof RepeaterDefinition repeaterDef)
            {
                if (repeaterDef.hasModelItem("common"))
                {
                    return true;
                }
            }
            parentItem = parentItem.getParent();
        }
        return false;
    }
    
    
    /**
     * Determines if this repeater is a repeater with a educational path
     * @param repeaterDefinition the repeater definition
     * @return true if the repeater contains data related to specific an education path
     */
    public boolean isRepeaterWithEducationalPath(RepeaterDefinition repeaterDefinition)
    {
        for (ModelItem childItem : repeaterDefinition.getModelItems())
        {
            if (EducationalPathRepositoryElementType.EDUCATIONAL_PATH_ELEMENT_TYPE_ID.equals(childItem.getType().getId()))
            {
                return true;
            }
        }
        return false;
    }
    
    /**
     * Browse entries of a repeater with educational path to get their values by educational path
     * @param content The content
     * @param repeaterDefinition The repeater definition
     * @return a Map of {@link EducationalPath} with related values by names
     */
    public Map<EducationalPath, Map<String, Object>> getValuesByEducationalPath(Content content, RepeaterDefinition repeaterDefinition)
    {
        Map<EducationalPath, Map<String, Object>> valuesByEducationalPath = new HashMap<>();
        
        String repeaterPath = repeaterDefinition.getPath();
        if (content.hasValue(repeaterPath) && isRepeaterWithEducationalPath(repeaterDefinition))
        {
            List< ? extends IndexableRepeaterEntry> entries = content.getRepeater(repeaterPath).getEntries();
            for (IndexableRepeaterEntry entry : entries)
            {
                EducationalPath path = null;
                Map<String, Object> values = new HashMap<>();
                for (ModelItem childItem : repeaterDefinition.getModelItems())
                {
                    if (EducationalPathRepositoryElementType.EDUCATIONAL_PATH_ELEMENT_TYPE_ID.equals(childItem.getType().getId()))
                    {
                        path = entry.getValue(childItem.getName());
                    }
                    else if (entry.hasValue(childItem.getName()))
                    {
                        values.put(childItem.getName(), entry.getValue(childItem.getName()));
                    }
                }
                
                if (path != null)
                {
                    valuesByEducationalPath.put(path, values);
                }
            }
        }
        return valuesByEducationalPath;
    }
    
    /**
     * Browse a repeater entries' values to reorganize values by educational path
     * @param repeaterValues The repeater entries' values
     * @param repeaterDefinition The repeater definition
     * @return a Map of {@link EducationalPath} with related values by names
     */
    public Map<EducationalPath, Map<String, Object>> getValuesByEducationalPath(List<Map<String, Object>> repeaterValues, RepeaterDefinition repeaterDefinition)
    {
        Map<EducationalPath, Map<String, Object>> valuesByEducationalPath = new HashMap<>();
        
        for (Map<String, Object> entryValues : repeaterValues)
        {
            EducationalPath path = null;
            Map<String, Object> values = new HashMap<>();
            
            for (ModelItem childItem : repeaterDefinition.getModelItems())
            {
                if (entryValues.containsKey(childItem.getName()))
                {
                    Object value = entryValues.get(childItem.getName());
                    value = value instanceof SynchronizableValue synchronizableValue ? synchronizableValue.getLocalValue() : value;
                    
                    if (value instanceof EducationalPath educationPath)
                    {
                        path = educationPath;
                    }
                    else
                    {
                        values.put(childItem.getName(), value);
                    }
                }
            }
            
            if (path != null)
            {
                valuesByEducationalPath.put(path, values);
            }
        }
        
        return valuesByEducationalPath;
    }
    
    private void _triggerWorkflowAction(Content content, int actionId) throws WorkflowException
    {
        if (content instanceof WorkflowAwareContent)
        {
            // The content has already been modified by this function
            SynchronizationResult synchronizationResult = new SynchronizationResult();
            synchronizationResult.setHasChanged(true);

            Map<String, Object> parameters = new HashMap<>();
            parameters.put(ValidateContentFunction.IS_MAJOR, false);
            parameters.put(EditContentFunction.SYNCHRONIZATION_RESULT, synchronizationResult);
            parameters.put(EditContentFunction.QUIT, true);

            Map<String, Object> inputs = new HashMap<>();
            inputs.put(AbstractWorkflowComponent.CONTEXT_PARAMETERS_KEY, parameters);
            
            // Do not check right
            inputs.put(CheckRightsCondition.FORCE, true);
            
            _workflowHelper.doAction((WorkflowAwareContent) content, actionId, inputs);
        }
    }
    
    private Pair<ProgramItem, String> _resolveEntryPath(Node entryNode) throws RepositoryException
    {
        List<String> entryPath = new ArrayList<>();
        
        String entryPos = StringUtils.substringAfter(entryNode.getName(), RepositoryConstants.NAMESPACE_PREFIX + ":");
        Node repeaterNode = entryNode.getParent();
        String repeaterName = StringUtils.substringAfter(repeaterNode.getName(), RepositoryConstants.NAMESPACE_PREFIX + ":");
        
        entryPath.add(repeaterName + "[" + entryPos + "]");
        
        Node currentNode = repeaterNode.getParent();
        
        while (!currentNode.isNodeType("ametys:content"))
        {
            String currentNodeName = StringUtils.substringAfter(currentNode.getName(), RepositoryConstants.NAMESPACE_PREFIX + ":");
            Node parentNode = currentNode.getParent();
            
            if (_isRepeaterNode(parentNode, currentNodeName))
            {
                entryPath.add(StringUtils.substringAfter(parentNode.getName(), RepositoryConstants.NAMESPACE_PREFIX + ":") + "[" + currentNodeName + "]");
                currentNode = parentNode.getParent();
            }
            else
            {
                entryPath.add(currentNodeName);
                currentNode = parentNode;
            }
        }
        
        ProgramItem programItem = _resolver.resolve(currentNode, false);
        
        Collections.reverse(entryPath);
        return Pair.of(programItem, StringUtils.join(entryPath, "/"));
    }
    
    private boolean _isRepeaterNode(Node node, String childName) throws RepositoryException 
    {
        return node.isNodeType("ametys:repeater") 
                || node.isNodeType("ametys:compositeMetadata") && StringUtils.isNumeric(childName); // legacy mode : old repeater nodes are of type ametys:compositeMetadata
    }
    
    private NodeIterator _getEducationalPathNodes(String... programItemIds) throws InvalidQueryException, RepositoryException
    {
        List<Expression> exprs = new ArrayList<>();
        for (String programItemId : programItemIds)
        {
            exprs.add(new StringExpression(EducationalPath.PATH_SEGMENTS_IDENTIFIER, Operator.EQ, programItemId));
        }
        
        String xPathQuery = QueryHelper.getXPathQuery(null, EducationalPathRepositoryElementType.EDUCATIONAL_PATH_NODETYPE, new AndExpression(exprs.toArray(Expression[]::new)));
        
        Session session = _repository.login();
        Query query = session.getWorkspace().getQueryManager().createQuery(xPathQuery, Query.XPATH);
        
        return query.execute().getNodes();
    }
    
    /**
     * Record for a {@link EducationalPath} references
     * @param programItem The program item holding the educational path
     * @param repeaterEntryPath The path of repeater entry holding the educational path
     * @param attributeName The name of attribute holding the educational path
     * @param value The value of the educational path
     */
    public record EducationalPathReference(ProgramItem programItem, String repeaterEntryPath, String attributeName, EducationalPath value) { /* empty */ }
}
