/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.data;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.lang3.StringUtils;

import org.ametys.odf.ProgramItem;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * Class representing an educational path of a {@link ProgramItem}
 */
public class EducationalPath
{
    /** The program items identifier */
    public static final String PATH_SEGMENTS_IDENTIFIER = "programItemIds";
    
    /** The separator for path segment */
    public static final String PATH_SEGMENT_SEPARATOR = ";";
    
    private List<String> _programItemIds;
    private List<ProgramItem> _programItems;

    /**
     * Constructs a EducationalPath
     * @param programItemIds the ids of program items 
     */
    EducationalPath(List<String> programItemIds)
    {
        _programItemIds = programItemIds;
    }
    
    /**
     * Constructs a EducationalPath
     * @param programItemsIds the ids of program items
     */
    EducationalPath(String[] programItemsIds)
    {
        _programItemIds = Arrays.asList(programItemsIds);
    }
    
    /**
     * Constructs a EducationalPath
     * @param programItems the program items
     */
    EducationalPath(ProgramItem[] programItems)
    {
        _programItems = Arrays.asList(programItems);
    }
    
    /**
     * Build a educational path
     * @param programItemIds the ids of program items
     * @return the educational path
     */
    public static EducationalPath of(String... programItemIds)
    {
        return new EducationalPath(programItemIds);
    }
    
    /**
     * Build a educational path from a parent education path
     * @param parentPath The parent educational path
     * @param programItemIds the ids of program items
     * @return the educational path
     */
    @SuppressWarnings("unchecked")
    public static EducationalPath of(EducationalPath parentPath, String... programItemIds)
    {
        List<String> parentProgramItemIds = new ArrayList(parentPath.getProgramItemIds());
        parentProgramItemIds.addAll(Arrays.asList(programItemIds));
        return new EducationalPath(parentProgramItemIds);
    }
    
    /**
     * Build a educational path from a parent education path
     * @param parentPath The parent educational path
     * @param programItem the program items
     * @return the educational path
     */
    public static EducationalPath of(EducationalPath parentPath, ProgramItem... programItem)
    {
        List<ProgramItem> parentProgramItems = parentPath._programItems;
        if (parentProgramItems != null)
        {
            List<ProgramItem> programItems = new ArrayList<>(parentProgramItems);
            programItems.addAll(Arrays.asList(programItem));
            return new EducationalPath(programItems.toArray(ProgramItem[]::new));
        }
        else
        {
            List<String> parentProgramItemIds = new ArrayList<>(parentPath.getProgramItemIds());
            parentProgramItemIds.addAll(Stream.of(programItem).map(ProgramItem::getId).collect(Collectors.toList()));
            return new EducationalPath(parentProgramItemIds);
        }
    }
    
    /**
     * Build a educational path
     * @param programItems the program items
     * @return the educational path
     */
    public static EducationalPath of(ProgramItem... programItems)
    {
        return new EducationalPath(programItems);
    }
    
    /**
     * Retrieves the program item ids that are the segment of this path
     * @return the program item ids
     */
    public List<String> getProgramItemIds()
    {
        if (_programItemIds != null)
        {
            return _programItemIds;
        }
        else
        {
            return _programItems.stream().map(ProgramItem::getId).collect(Collectors.toList());
        }
    }
    
    /**
     * Get {@link ProgramItem}s that composed this educational path as an unmodifiable list
     * @param resolver the ametys object resolver
     * @return the {@link ProgramItem}s of this path
     */
    public List<ProgramItem> getProgramItems(AmetysObjectResolver resolver)
    {
        if (_programItems != null)
        {
            return _programItems;
        }
        else
        {
            return resolveProgramItems(resolver).toList();
        }
    }
    
    /**
     * Resolve {@link ProgramItem}s that composed this educational path 
     * @param resolver the ametys object resolver
     * @return the {@link ProgramItem}s of this path
     */
    public Stream<ProgramItem> resolveProgramItems(AmetysObjectResolver resolver)
    {
        if (_programItems != null)
        {
            return _programItems.stream();
        }
        else
        {
            return _programItemIds
                    .stream()
                    .map(id -> (ProgramItem) resolver.resolveById(id));
        }
    }
    
    @Override
    public String toString()
    {
        return StringUtils.join(getProgramItemIds(), PATH_SEGMENT_SEPARATOR);
    }
    
    @Override
    public int hashCode()
    {
        return Objects.hash(getProgramItemIds());
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }
        if (!(obj instanceof EducationalPath other))
        {
            return false;
        }
        
        return Objects.equals(getProgramItemIds(), other.getProgramItemIds());
    }
}
