/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.oai;

import java.io.IOException;
import java.util.Collection;
import java.util.Enumeration;

import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang.StringUtils;
import org.xml.sax.SAXException;

/**
 * Base class for all OAI verb responses.
 */
public abstract class AbstractOAIVerbGenerator extends AbstractOAIResponseGenerator
{
    @Override
    protected final void doGenerate() throws IOException, SAXException, ProcessingException
    {
        if (_hasBadArguments(getRequiredParameters(), getAllowedParameters()))
        {
            XMLUtils.createElement(contentHandler, "request", getURL());
            generateError("badArgument", "The request includes illegal arguments, is missing required arguments, includes a repeated argument, or values for arguments have an illegal syntax.");
        }
        else
        {
            Request request = ObjectModelHelper.getRequest(objectModel);
            
            AttributesImpl params = new AttributesImpl();
            
            Enumeration paramsEnum = request.getParameterNames();
            while (paramsEnum.hasMoreElements())
            {
                String name = (String) paramsEnum.nextElement();
                params.addCDATAAttribute(name, request.getParameter(name));
            }
            
            XMLUtils.createElement(contentHandler, "request", params, getURL());

            generateVerb();
        }
    }
    
    /**
     * Returns a Collection af all mandatory parameters.
     * @return a Collection af all mandatory parameters.
     */
    protected abstract Collection<String> getRequiredParameters();

    /**
     * Returns a Collection af all possible parameters.
     * @return a Collection af all possible parameters.
     */
    protected abstract Collection<String> getAllowedParameters();
    
    /**
     * Generates SAX events for the current verb.
     * @throws IOException if an I/O error occurs
     * @throws SAXException if an error occurs
     * @throws ProcessingException if an error occurs
     */
    protected abstract void generateVerb() throws IOException, SAXException, ProcessingException;
    
    private boolean _hasBadArguments(Collection<String> requiredParameters, Collection<String> allowedParameters)
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        for (String requiredParam : requiredParameters)
        {
            if (StringUtils.isEmpty(request.getParameter(requiredParam)))
            {
                return true;
            }
        }
        
        Enumeration params = request.getParameterNames();
        while (params.hasMoreElements())
        {
            String name = (String) params.nextElement();
            if (!allowedParameters.contains(name) || request.getParameterValues(name).length > 1)
            {
                return true;
            }
        }
        
        return false;
    }
}
