/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.oai;

import java.time.ZoneOffset;
import java.time.ZonedDateTime;

import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.core.util.DateUtils;
import org.ametys.odf.cdmfr.ExportCDMfrManager;
import org.ametys.odf.program.Program;

/**
 * Helper for handling OAI records/
 */
public final class RecordHelper
{
    private RecordHelper()
    {
        // empty
    }
   
    /**
     * Generates SAX events representing a Program as an OAI record.
     * @param program the {@link Program}.
     * @param contentHandler the {@link ContentHandler}.
     * @param metadataPrefix "cdm" or "oai_dc".
     * @param exportCDMfrManager the {@link ExportCDMfrManager}.
     * @throws SAXException if an error occurs.
     */
    public static void saxProgram(Program program, ContentHandler contentHandler, String metadataPrefix, ExportCDMfrManager exportCDMfrManager) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "record");
        
        XMLUtils.startElement(contentHandler, "header");
        
        XMLUtils.createElement(contentHandler, "identifier", program.getCDMId());
        
        ZonedDateTime date = program.getLastModified();
        String lastModified = DateUtils.zonedDateTimeToString(date, ZoneOffset.UTC, "yyyy-MM-dd'T'HH:mm:ssXXX");
        XMLUtils.createElement(contentHandler, "datestamp", lastModified);
        
        XMLUtils.endElement(contentHandler, "header");
        
        XMLUtils.startElement(contentHandler, "metadata");
        
        if (metadataPrefix.equals("cdm"))
        {
            // CDM-fr
            contentHandler.startPrefixMapping("", "http://cdm-fr.fr/2011/CDM");
            contentHandler.startPrefixMapping("cdmfr", "http://cdm-fr.fr/2011/CDM-frSchema");
            contentHandler.startPrefixMapping("ametys-cdm", "http://www.ametys.org/cdm/1.0");
            contentHandler.startPrefixMapping("xhtml", "http://www.w3.org/1999/xhtml");
            
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("http://www.w3.org/2001/XMLSchema-instance", "schemaLocation", "xsi:schemaLocation", "http://cdm-fr.fr/2011/CDM http://cdm-fr.fr/2011/schemas/CDMFR.xsd");
            attrs.addCDATAAttribute("profile", "CDM-fr");
            attrs.addCDATAAttribute("language", program.getLanguage());
            XMLUtils.startElement(contentHandler, "CDM", attrs);

            exportCDMfrManager.saxHabilitation(contentHandler, program);
            exportCDMfrManager.program2CDM(contentHandler, program);
            
            XMLUtils.endElement(contentHandler, "CDM");
        }
        else
        {
            // Dublin core
            contentHandler.startPrefixMapping("oai_dc", "http://www.openarchives.org/OAI/2.0/oai_dc/");
            contentHandler.startPrefixMapping("dc", "http://purl.org/dc/elements/1.1/");
            
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("http://www.w3.org/2001/XMLSchema-instance", "schemaLocation", "xsi:schemaLocation", "http://www.openarchives.org/OAI/2.0/oai_dc/ http://www.openarchives.org/OAI/2.0/oai_dc.xsd");
            contentHandler.startElement("http://www.openarchives.org/OAI/2.0/oai_dc/", "dc", "oai_dc:dc", attrs);
            
            contentHandler.startElement("http://purl.org/dc/elements/1.1/", "title", "dc:title", new AttributesImpl());
            String title = program.getTitle();
            contentHandler.characters(title.toCharArray(), 0, title.length());
            contentHandler.endElement("http://purl.org/dc/elements/1.1/", "title", "dc:title");
            
            contentHandler.startElement("http://purl.org/dc/elements/1.1/", "creator", "dc:creator", new AttributesImpl());
            String creator = "Ametys";
            contentHandler.characters(creator.toCharArray(), 0, creator.length());
            contentHandler.endElement("http://purl.org/dc/elements/1.1/", "creator", "dc:creator");
            
            contentHandler.startElement("http://purl.org/dc/elements/1.1/", "date", "dc:date", new AttributesImpl());
            contentHandler.characters(lastModified.toCharArray(), 0, lastModified.length());
            contentHandler.endElement("http://purl.org/dc/elements/1.1/", "date", "dc:date");
            
            contentHandler.endElement("http://www.openarchives.org/OAI/2.0/oai_dc/", "dc", "oai_dc:dc");
        }
        
        XMLUtils.endElement(contentHandler, "metadata");
        
        XMLUtils.endElement(contentHandler, "record");
    }
}
