/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.observation.skill;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.holder.group.ModifiableIndexableRepeater;
import org.ametys.cms.data.holder.group.ModifiableIndexableRepeaterEntry;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.program.Program;

/**
 * Observer to remove tranvsersal skills from children courses if they have been removed from the parent Program.
 * When the modified content is a Program, remove the tranvsersal macro skills, that got removed from the program, from the children courses
 */
public class UpdateProgramSkillStep2Observer extends AbstractSkillsStepObserver
{
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
    }
    
    @Override
    protected String getSupportedEventId()
    {
        return ObservationConstants.EVENT_CONTENT_MODIFIED;
    }
    
    @Override
    protected boolean supportsContent(Content content)
    {
        // Only support Programs
        return content instanceof Program;
    }
    
    @Override
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Program program = (Program) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        
        // When the content is modified, we need to check if the transversal skills are still linked to the content
        ContentValue[] previousTransversalSkills = _getRequestAttribute("previousTransversalSkills", program.getId());
        if (previousTransversalSkills != null)
        {
            ContentValue[] currentTransversalSkills = program.getValue(Program.TRANSVERSAL_SKILLS, false, new ContentValue[0]);

            // Compare the previous transversal skills with the current ones to retrieve the skills which are not linked to the program anymore and remove them from children courses
            ContentValue[] transversalSkillsToDeleteContentValue = getSkillsToDelete(previousTransversalSkills, currentTransversalSkills);
            if (transversalSkillsToDeleteContentValue != null)
            {
                List<String> microSkillsToDeleteIds = Arrays.stream(transversalSkillsToDeleteContentValue)
                    .map(ContentValue::getContentIfExists)
                    .flatMap(Optional::stream)
                    .flatMap(skill -> Arrays.stream((ContentValue[]) skill.getValue("microSkills")))
                    .map(ContentValue::getContentId)
                    .toList();
                
                if (!microSkillsToDeleteIds.isEmpty())
                {
                    // Remove the orphan transversal skills from children courses
                    _removeTransversalSkillsRecursively(program, microSkillsToDeleteIds);
                }
            }
        }
    }

    private void _removeTransversalSkillsRecursively(Program program, List<String> microSkillsToDeleteIds)
    {
        // Remove the blocking micro skills that where from removed macro skills
        ContentValue[] blockingSkills = program.getValue(Program.BLOCKING_SKILLS, false, new ContentValue[0]);
        ContentValue[] blockingSkillsToSet = Arrays.stream(blockingSkills)
            // Only retrieve the microSkills that did not get removed from the program
            .filter(microSkill -> !microSkillsToDeleteIds.contains(microSkill.getContentId()))
            .toArray(ContentValue[]::new);
        
        if (blockingSkillsToSet.length == 0)
        {
            program.removeValue(Program.BLOCKING_SKILLS);
        }
        else
        {
            program.setValue(Program.BLOCKING_SKILLS, blockingSkillsToSet);
        }
        program.saveChanges();
        
        _removeTransversalSkillsRecursively(program, program, microSkillsToDeleteIds);
    }
    
    private void _removeTransversalSkillsRecursively(Program program, ProgramItem programItem, List<String> microSkillsToDeleteIds)
    {
        if (programItem instanceof Course course)
        {
            if (course.hasValue(Course.ACQUIRED_MICRO_SKILLS))
            {
                ModifiableIndexableRepeater repeater = course.getRepeater(Course.ACQUIRED_MICRO_SKILLS);
                ModifiableIndexableRepeaterEntry programEntry = repeater.getEntries().stream()
                    .filter(entry -> program.getId().equals(((ContentValue) entry.getValue(Course.ACQUIRED_MICRO_SKILLS_PROGRAM)).getContentId()))
                    .findFirst()
                    .orElse(null);
                if (programEntry != null)
                {
                    ContentValue[] microSkills = programEntry.getValue(Course.ACQUIRED_MICRO_SKILLS_SKILLS, false, null);
                    
                    if (microSkills != null)
                    {
                        ContentValue[] microSkillsToSet = Arrays.stream(microSkills)
                            // Only retrieve the microSkills that did not get removed from the program
                            .filter(microSkill -> !microSkillsToDeleteIds.contains(microSkill.getContentId()))
                            .toArray(ContentValue[]::new);
                        
                        // If there are no micro skills for this program in this course anymore, remove the whole entry
                        if (microSkillsToSet.length == 0)
                        {
                            repeater.removeEntry(programEntry.getPosition());
                        }
                        // If there are still some micro skills, set them
                        else
                        {
                            programEntry.setValue(Course.ACQUIRED_MICRO_SKILLS_SKILLS, microSkillsToSet);
                        }
                        
                        course.saveChanges();
                    }
                }
            }
        }
        
        List<ProgramItem> children = _odfHelper.getChildProgramItems(programItem);
        for (ProgramItem childProgramItem : children)
        {
            _removeTransversalSkillsRecursively(program, childProgramItem, microSkillsToDeleteIds);
        }
    }
}
