/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.workflow.task;

import java.util.List;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.cms.workflow.WorkflowTasksComponent;
import org.ametys.core.user.User;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.catalog.Catalog;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.plugins.repository.query.expression.Expression;

/**
 * Override WorkflowTasksComponent to target ODF contents
 */
public abstract class AbstractOdfWorkflowTasksComponent extends WorkflowTasksComponent
{
    /** Catalogs manager */
    protected CatalogsManager _catalogsManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _catalogsManager = (CatalogsManager) manager.lookup(CatalogsManager.ROLE);
    }
    
    @Override
    protected List<Expression> _getContentsAndExpressions(TaskStep step, User user)
    {
        List<Expression> exprs = super._getContentsAndExpressions(step, user);
        
        exprs.add(getContentTypeExpression());
        
        return exprs;
    }

    @Override
    protected void _saxAdditionalData(ContentHandler ch, Content content, Task task) throws SAXException
    {
        super._saxAdditionalData(ch, content, task);
        
        if (content instanceof ProgramItem)
        {
            ProgramItem programItem = (ProgramItem) content;
            
            // Code
            XMLUtils.createElement(ch, "code", programItem.getCode());
            XMLUtils.createElement(ch, "displayCode", programItem.getDisplayCode());
            
            // Catalog
            Catalog catalog = Optional.of(programItem)
                .map(ProgramItem::getCatalog)
                .map(_catalogsManager::getCatalog)
                .orElse(null);
            if (catalog != null)
            {
                AttributesImpl attrs = new AttributesImpl();
                attrs.addCDATAAttribute("id", catalog.getId());
                XMLUtils.createElement(ch, "catalog", attrs, catalog.getTitle());
            }
        }
    }
    
    /**
     * Get the content type expression concerned by this component
     * @return the content type expression
     */
    protected abstract Expression getContentTypeExpression ();
}
