/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.catalog;

import java.util.HashMap;
import java.util.Map;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.odf.translation.TranslationHelper;

/**
 * This updater updates the translations of created contents after the full catalog copy
 *
 */
public class TranslationUpdater implements CopyCatalogUpdater
{
    @Override
    public void updateContents(String initialCatalogName, String newCatalogName, Map<Content, Content> copiedContents, Content targetParentContent)
    {
        for (Content initialContent : copiedContents.keySet())
        {
            // Get the translations of initial content
            Map<String, String> translations = TranslationHelper.getTranslations(initialContent);
            
            if (!translations.isEmpty())
            {
                Map<String, String> updatedTranslations = new HashMap<>();
                
                for (String lang : translations.keySet())
                {
                    // For each translation, get its copy in the new catalog
                    String translatedContentId = translations.get(lang);
                    
                    copiedContents.entrySet()
                                  .stream()
                                  .filter(entry -> translatedContentId.equals(entry.getKey().getId()))
                                  .findFirst()
                                  .map(Map.Entry::getValue)
                                  .map(Content::getId)
                                  .ifPresent(copyId -> updatedTranslations.put(lang, copyId));
                }
                
                // Update the translations for created content
                Content createdContent = copiedContents.get(initialContent);
                if (createdContent instanceof ModifiableContent modifiableCreatedContent)
                {
                    TranslationHelper.setTranslations(modifiableCreatedContent, updatedTranslations);
                }
            }
        }
    }
}
