/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.export;

import java.io.IOException;
import java.time.ZonedDateTime;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.util.DateUtils;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Program;
import org.ametys.odf.tree.OdfClassificationHandler;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectIterator;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.version.VersionAwareAmetysObject;
import org.ametys.runtime.model.View;

/**
 * Generate the ODF structure with 2 levels (metadata), the catalog and the lang. It's possible to determine a metadataset to sax data.
 * 
 * You should call this generator with the following parameters :
 *  - catalog : identifier of the catalog
 *  - lang : language code (fr, en, etc.)
 *  - level1 : name of the attribute for the first level
 *  - level2 : name of the attribute for the second level
 *  - metadataSet (optional) : name of the view to sax values
 * 
 */
public class ExportCatalogByLevelsGenerator extends ServiceableGenerator
{
    /** The AmetysObject resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The content type helper */
    protected ContentTypesHelper _contentTypesHelper;
    
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    /** The ODF classification handler */
    protected OdfClassificationHandler _odfClassificationHandler;
    
    /** The helper for reference tables*/
    protected OdfReferenceTableHelper _odfRefTableHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _contentTypesHelper = (ContentTypesHelper) smanager.lookup(ContentTypesHelper.ROLE);
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
        _odfClassificationHandler = (OdfClassificationHandler) smanager.lookup(OdfClassificationHandler.ROLE);
        _odfRefTableHelper = (OdfReferenceTableHelper) smanager.lookup(OdfReferenceTableHelper.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "Ametys-ODF");
        
        /* Test parameters */
        Map<String, String> wsParameters = new HashMap<>();
        if (getParameters(wsParameters))
        {
            /* Build and SAX ODF Structure */
            AmetysObjectIterable<Program> programs = getConcernedPrograms(wsParameters);
            
            // Switch to version label if not empty
            String versionLabel = parameters.getParameter("versionLabel", null);
            if (StringUtils.isNotBlank(versionLabel))
            {
                // wrap the current iterable into a iteratable which will filtered programs by version label
                programs = new FilteredByVersionLabelIterable<>(programs, versionLabel);
            }
            
            Map<String, Map<String, Collection<Program>>> odfStructure =  _odfClassificationHandler.organizeProgramsByLevels(programs, wsParameters.get("level1"), wsParameters.get("level2"));

            String level1 = wsParameters.get("level1");
            String level2 = wsParameters.get("level2");
            String viewName = wsParameters.get("metadataSet");
            
            for (String level1Value : odfStructure.keySet())
            {
                AttributesImpl attrs = new AttributesImpl();
                _addLevelAttributes(attrs, level1Value, wsParameters.get("lang"));
                XMLUtils.startElement(contentHandler, level1, attrs);
                
                Map<String, Collection<Program>> level2Values = odfStructure.get(level1Value);
                for (String level2Value : level2Values.keySet())
                {
                    attrs.clear();
                    _addLevelAttributes(attrs, level2Value, wsParameters.get("lang"));
                    
                    XMLUtils.startElement(contentHandler, level2, attrs);
                    for (Program program : level2Values.get(level2Value))
                    {
                        _saxStructure(program, program, viewName, wsParameters);
                    }
                    XMLUtils.endElement(contentHandler, level2);
                }
                
                XMLUtils.endElement(contentHandler, level1);
            }
        }
        
        XMLUtils.endElement(contentHandler, "Ametys-ODF");
        contentHandler.endDocument();
    }
    
    /**
     * Add attributes for classification level
     * @param attrs The XML attributes
     * @param value The level's value
     * @param lang The language
     */
    protected void _addLevelAttributes(AttributesImpl attrs, String value, String lang)
    {
        if (_resolver.hasAmetysObjectForId(value))
        {
            Content content = _resolver.resolveById(value);
            
            if (_odfRefTableHelper.isTableReferenceEntry(content))
            {
                OdfReferenceTableEntry entry = new OdfReferenceTableEntry(content);
                attrs.addCDATAAttribute("id", entry.getId());
                attrs.addCDATAAttribute("code", entry.getCode());
                attrs.addCDATAAttribute("title", entry.getLabel(lang));
            }
            else
            {
                attrs.addCDATAAttribute("id", content.getId());
                attrs.addCDATAAttribute("title", content.getTitle());
            }
        }
        else
        {
            attrs.addCDATAAttribute("value", value);
        }
    }
    
    /**
     * Get the parameters from the request and test it.
     * @param wsParameters Map of parameters to fill
     * @return false if a parameter is missing or something going wrong with the parameters, otherwise true
     * @throws SAXException if an error occured
     */
    protected boolean getParameters(Map<String, String> wsParameters) throws SAXException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);

        String catalog = getParameter(request, "catalog");
        String lang = getParameter(request, "lang");
        String level1 = getParameter(request, "level1");
        String level2 = getParameter(request, "level2");
        
        boolean isValidLevelParameters = true;
        if (level1 != null && !_odfClassificationHandler.isEligibleMetadataForLevel(level1, true))
        {
            XMLUtils.createElement(contentHandler, "error", "The metadata " + level1 + " is not an eligible metadata for the export");
            isValidLevelParameters = false;
        }
        
        if (level2 != null && !_odfClassificationHandler.isEligibleMetadataForLevel(level2, true))
        {
            XMLUtils.createElement(contentHandler, "error", "The metadata " + level2 + " is not an eligible metadata for the export");
            isValidLevelParameters = false;
        }
        
        wsParameters.put("catalog", catalog);
        wsParameters.put("lang", lang);
        wsParameters.put("level1", level1);
        wsParameters.put("level2", level2);
        
        String metadataSet = request.getParameter("metadataSet");
        if (StringUtils.isBlank(metadataSet))
        {
            metadataSet = "main";
        }
        wsParameters.put("metadataSet", metadataSet);
        
        return catalog != null && lang != null && isValidLevelParameters && level1 != null && level2 != null;
    }
    
    /**
     * Get the parameter from the request and test if it's not null or blank.
     * Sax an error if the parameter is missing or empty.
     * @param request The request
     * @param parameterName The parameter name
     * @return null when the parameter is missing or empty, otherwise the parameter value
     * @throws SAXException if an error occured
     */
    protected String getParameter(Request request, String parameterName) throws SAXException
    {
        String parameterValue = parameters.getParameter(parameterName, request.getParameter(parameterName));

        if (StringUtils.isBlank(parameterValue))
        {
            XMLUtils.createElement(contentHandler, "error", "Missing parameter (cannot be empty) : " + parameterName);
            parameterValue = null;
        }
        
        return parameterValue;
    }
    
    /**
     * Get the programs to SAX.
     * @param wsParameters Parameters of the web service
     * @return A Collection of programs
     */
    protected AmetysObjectIterable<Program> getConcernedPrograms(Map<String, String> wsParameters)
    {
        return _odfClassificationHandler.getPrograms(wsParameters.get("catalog"), wsParameters.get("lang"), wsParameters.get("level1"), null, wsParameters.get("level2"), null, null, null, null);
    }
    
    /**
     * Generates SAX events for the structure of the parentProgram by exploring its children and generating SAX events for the passed view.
     * @param parentProgram Initial program
     * @param programItem Part of the program to explore
     * @param viewName Name of the view to SAX
     * @param wsParameters Parameters of the web service
     * @throws AmetysRepositoryException if an error occurred
     * @throws SAXException if an error occurred
     * @throws IOException if an error occurred
     */
    private void _saxStructure(Program parentProgram, ProgramItem programItem, String viewName, Map<String, String> wsParameters) throws AmetysRepositoryException, SAXException, IOException
    {
        if (programItem instanceof AbstractProgram || programItem instanceof Course)
        {
            Content content = (Content) programItem;
            
            String contentType = content.getTypes()[0];
            contentType = contentType.substring(contentType.lastIndexOf(".") + 1);

            AttributesImpl attrs = getContentAttributes(programItem, parentProgram, wsParameters);
            
            XMLUtils.startElement(contentHandler, contentType, attrs);
    
            /* SAX attributes for flat level */
            View view = _contentTypesHelper.getView(viewName, content.getTypes(), content.getMixinTypes());
            content.dataToSAX(contentHandler, view);
            
            /* SAX structure of the ProgramItem */
            _saxChildren(parentProgram, programItem, viewName, wsParameters);
            
            XMLUtils.endElement(contentHandler, contentType);
        }
        else
        {
            /* SAX structure of the ProgramItem */
            _saxChildren(parentProgram, programItem, viewName, wsParameters);
        }
    }
    
    /**
     * Explore and sax children of the passed program item.
     * @param parentProgram Initial program
     * @param programItem Part of the program to explore
     * @param viewName Name of the view to SAX
     * @param wsParameters Parameters of the web service
     * @throws AmetysRepositoryException if an error occurred
     * @throws SAXException if an error occurred
     * @throws IOException if an error occurred
     */
    private void _saxChildren(Program parentProgram, ProgramItem programItem, String viewName, Map<String, String> wsParameters) throws AmetysRepositoryException, SAXException, IOException
    {
        for (ProgramItem child : _odfHelper.getChildProgramItems(programItem))
        {
            _saxStructure(parentProgram, child, viewName, wsParameters);
        }
    }
    
    /**
     * Get attributes for the current saxed content (title, id, etc.).
     * @param programItem Part of the program to get attributes
     * @param parentProgram Initial program
     * @param wsParameters Parameters of the web service
     * @return The attributes to sax
     */
    protected AttributesImpl getContentAttributes(ProgramItem programItem, Program parentProgram, Map<String, String> wsParameters)
    {
        Content content = (Content) programItem;
        
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("title", content.getTitle());
        attrs.addCDATAAttribute("id", content.getId());
        ZonedDateTime lastValidated = content.getLastValidationDate();
        if (lastValidated != null)
        {
            attrs.addCDATAAttribute("lastValidated", DateUtils.zonedDateTimeToString(lastValidated));
        }
        return attrs;
    }
    
    class FilteredByVersionLabelIterable<P extends VersionAwareAmetysObject> implements AmetysObjectIterable<P>
    {
        private AmetysObjectIterable<P> _initialIterable;
        private String _versionLabel;
        
        /**
         * Creates a {@link AmetysObjectIterable} which will filter and get elements with given version label
         * @param it the initial {@link AmetysObjectIterable}s
         * @param versionLabel The version label to filter by
         */
        public FilteredByVersionLabelIterable(AmetysObjectIterable<P> it, String versionLabel)
        {
            _initialIterable = it;
            _versionLabel = versionLabel;
        }
        
        public long getSize()
        {
            return -1;
        }
        
        public AmetysObjectIterator<P> iterator()
        {
            return new FilteredByVersionLabelIterator(_initialIterable.iterator(), _initialIterable.getSize(), _versionLabel);
        }
        
        public void close()
        {
            // nothing to do
        }
        
        class FilteredByVersionLabelIterator implements AmetysObjectIterator<P>
        {
            private long _invalids;
            private Iterator<P> _it;
            private int _pos;
            private long _size;
            private P _nextObject;
            private String _label;
            
            public FilteredByVersionLabelIterator(Iterator<P> it, long size, String label)
            {
                _it = it;
                _size = size;
                _invalids = 0;
                _label = label;
            }
            
            public boolean hasNext()
            {
                // Prefetch the next object
                if (_nextObject == null)
                {
                    while (_it.hasNext())
                    {
                        P next = _it.next();
                        try
                        {
                            String[] allLabels = next.getAllLabels();
                            if (ArrayUtils.contains(allLabels, _label))
                            {
                                next.switchToLabel(_label);
                                _nextObject = next;
                                return true;
                            }
                            else
                            {
                                // Go to next element
                                _invalids++;
                            }
                        }
                        catch (UnknownAmetysObjectException e)
                        {
                            // Go to next element
                            _invalids++;
                        }
                    }
                    return false;
                }
                return true;
            }

            public P next()
            {
                if (!hasNext())
                {
                    throw new NoSuchElementException();
                }
                
                try
                {
                    _pos++;
                    return _nextObject;
                }
                finally
                {
                    _nextObject = null;
                }
            }

            public long getSize()
            {
                return _size - _invalids;
            }
            
            public long getPosition()
            {
                return _pos;
            }
        }
    }
}
