/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.export.pdf;

import java.io.IOException;
import java.net.MalformedURLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.parameters.ParameterException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentAttributeDefinition;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.LanguageExpression;
import org.ametys.core.util.IgnoreRootHandler;
import org.ametys.core.util.LambdaUtils;
import org.ametys.odf.NoLiveVersionException;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.catalog.Catalog;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.ProgramPart;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.program.SubProgramFactory;
import org.ametys.odf.schedulable.CatalogPDFExportSchedulable;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CollectionIterable;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.SortCriteria;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.model.View;

/**
 * Generator producing the SAX events for the catalogue summary 
 */ 
public class FOProgramsGenerator extends ServiceableGenerator
{
    private static final String __EXPORT_MODE = "catalog";
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The content type extension point */
    protected ContentTypeExtensionPoint _ctypeEP;
    /** The ODf helper */
    protected ODFHelper _odfHelper;
    /** The ODf enumeration helper */
    protected OdfReferenceTableHelper _odfTableRefHelper;
    /** The catalog manager */
    protected CatalogsManager _catalogManager;
    /** The query helper */
    protected org.ametys.plugins.queriesdirectory.QueryHelper _queryHelper;
    
    @Override
    public void service(ServiceManager sManager) throws ServiceException
    {
        super.service(sManager);
        _resolver = (AmetysObjectResolver) sManager.lookup(AmetysObjectResolver.ROLE);
        _ctypeEP = (ContentTypeExtensionPoint) sManager.lookup(ContentTypeExtensionPoint.ROLE);
        _odfHelper = (ODFHelper) sManager.lookup(ODFHelper.ROLE);
        _odfTableRefHelper = (OdfReferenceTableHelper) sManager.lookup(OdfReferenceTableHelper.ROLE);
        _catalogManager = (CatalogsManager) sManager.lookup(CatalogsManager.ROLE);
        _queryHelper = (org.ametys.plugins.queriesdirectory.QueryHelper) sManager.lookup(org.ametys.plugins.queriesdirectory.QueryHelper.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "programs");
        
        Catalog catalog = null;
        if ("_default".equals(source))
        {
            catalog = _catalogManager.getDefaultCatalog();
        }
        else
        {
            catalog = _catalogManager.getCatalog(source);
        }
        
        if (catalog == null)
        {
            throw new IllegalArgumentException ("Failed to generated PDF of unknown catalog '" + source + "'");
        }

        String lang;
        try
        {
            lang = parameters.getParameter("lang");
        }
        catch (ParameterException e)
        {
            throw new IllegalArgumentException ("Missing lang parameter", e);
        }
        
        // Catalog
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("name", source);
        XMLUtils.createElement(contentHandler, "catalog", attrs, catalog.getTitle());

        Map parentContext = (Map) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        
        boolean queryMode = CatalogPDFExportSchedulable.MODE_QUERY.equals(parentContext.get(CatalogPDFExportSchedulable.JOBDATAMAP_MODE_KEY));
        
        // Orgunits
        List<OrgUnit> orgUnits = new ArrayList<>();
        if (parentContext.containsKey(CatalogPDFExportSchedulable.JOBDATAMAP_ORGUNIT_KEY))
        {
            Object[] ouIds = (Object[]) parentContext.get(CatalogPDFExportSchedulable.JOBDATAMAP_ORGUNIT_KEY);
            
            orgUnits = Arrays.stream(ouIds)
                .map(String.class::cast)
                .filter(StringUtils::isNotEmpty)
                .map(LambdaUtils.wrap(_resolver::<OrgUnit>resolveById))
                .collect(Collectors.toList());
        }
        
        for (OrgUnit orgUnit : orgUnits)
        {
            attrs.clear();
            attrs.addCDATAAttribute("id", orgUnit.getId());
            attrs.addCDATAAttribute("uaiCode", orgUnit.getUAICode());
            XMLUtils.createElement(contentHandler, "orgunit", attrs, orgUnit.getTitle());
        }
        
        // Degrees
        List<OdfReferenceTableEntry> degrees = new ArrayList<>();
        if (parentContext.containsKey(CatalogPDFExportSchedulable.JOBDATAMAP_DEGREE_KEY))
        {
            Object[] degreeIds = (Object[]) parentContext.get(CatalogPDFExportSchedulable.JOBDATAMAP_DEGREE_KEY);
            
            degrees = Arrays.stream(degreeIds)
                .map(String.class::cast)
                .filter(StringUtils::isNotEmpty)
                .map(_odfTableRefHelper::getItem)
                .collect(Collectors.toList());
        }
        
        for (OdfReferenceTableEntry degree : degrees)
        {
            attrs.clear();
            attrs.addCDATAAttribute("id", degree.getId());
            attrs.addCDATAAttribute("code", degree.getCode());
            attrs.addCDATAAttribute("order", String.valueOf(degree.getOrder()));
            XMLUtils.createElement(contentHandler, "degree", attrs, degree.getLabel(lang));
        }
        
        String queryId = (String) parentContext.get(CatalogPDFExportSchedulable.JOBDATAMAP_QUERY_KEY);
        boolean includeSubPrograms = (Boolean) parentContext.get(CatalogPDFExportSchedulable.JOBDATAMAP_INCLUDE_SUBPROGRAMS);
        
        Map<String, ContentAttributeDefinition> tableRefAttributeDefs = _odfTableRefHelper.getTableRefAttributeDefinitions(ProgramFactory.PROGRAM_CONTENT_TYPE);
        
        AmetysObjectIterable<Program> programs = queryMode ? _getPrograms(catalog.getName(), lang, queryId) : _getPrograms(catalog.getName(), lang, orgUnits, degrees);
        _saxPrograms(programs, includeSubPrograms);
        
        // SAX entries of table references
        XMLUtils.startElement(contentHandler, "enumerated-metadata");
        for (ContentAttributeDefinition attributeDef : tableRefAttributeDefs.values())
        {
            _odfTableRefHelper.saxItems(contentHandler, attributeDef, lang);
        }
        XMLUtils.endElement(contentHandler, "enumerated-metadata");
        
        XMLUtils.endElement(contentHandler, "programs");
        contentHandler.endDocument();
    }
    
    /**
     * Get programs from catalog, lang, orgunit and degree. Orgunit and degree can be null.
     * @param catalog The catalog
     * @param lang The content language
     * @param orgUnits The restricted orgunits. Can be empty to not filter by orgunits
     * @param degrees The restricted degrees. Can be empty to not filter by degrees
     * @return An iterable of programs corresponding to the query with catalog, lang, orgunit and degree.
     */
    protected AmetysObjectIterable<Program> _getPrograms(String catalog, String lang, List<OrgUnit> orgUnits, List<OdfReferenceTableEntry> degrees)
    {
        List<Expression> exprs = new ArrayList<>();
        exprs.add(new ContentTypeExpression(Operator.EQ, ProgramFactory.PROGRAM_CONTENT_TYPE));
        exprs.add(new StringExpression(ProgramItem.CATALOG, Operator.EQ, catalog));
        exprs.add(new LanguageExpression(Operator.EQ, lang));

        if (!degrees.isEmpty())
        {
            Expression[] degreeExprs = degrees.stream()
                .map(d -> new StringExpression(AbstractProgram.DEGREE, Operator.EQ, d.getId()))
                .toArray(Expression[]::new);
            
            exprs.add(new OrExpression(degreeExprs));
        }
        
        if (!orgUnits.isEmpty())
        {
            Expression[] ouExprs = orgUnits.stream()
                .map(_odfHelper::getSubOrgUnitIds)
                .flatMap(List::stream)
                .distinct()
                .map(orgunitId -> new StringExpression(ProgramItem.ORG_UNITS_REFERENCES, Operator.EQ, orgunitId))
                .toArray(Expression[]::new);
            
            exprs.add(new OrExpression(ouExprs));
        }
        
        Expression programsExpression = new AndExpression(exprs.toArray(Expression[]::new));
        
        SortCriteria sortCriteria = new SortCriteria();
        sortCriteria.addCriterion(Content.ATTRIBUTE_TITLE, true, true);
  
        String programsQuery = QueryHelper.getXPathQuery(null, ProgramFactory.PROGRAM_NODETYPE, programsExpression, sortCriteria);
        AmetysObjectIterable<Program> programs = _resolver.query(programsQuery);
        
        return programs;
    }
    
    /**
     * Get programs from catalog, lang and a query
     * @param catalog The catalog
     * @param lang The content language
     * @param queryId The query id
     * @return An iterable of programs corresponding to the query. Results are filtered 
     * @throws ProcessingException if failed to execute query
     */
    protected AmetysObjectIterable<Program> _getPrograms(String catalog, String lang, String queryId) throws ProcessingException
    {
        try
        {
            AmetysObjectIterable<Content> contents = _queryHelper.executeQuery(queryId);
            
            List<Program> programs = contents.stream()
                .filter(Program.class::isInstance)
                .map(Program.class::cast)
                .filter(p -> catalog.equals(p.getCatalog()))
                .filter(p -> lang.equals(p.getLanguage()))
                .toList();
                
            return new CollectionIterable<>(programs);
        }
        catch (Exception e)
        {
            throw new ProcessingException("Failed to execute query '" + queryId + "' to generate PDF catalog", e);
        }
    }
    
    /**
     * Sax programs
     * @param programs the programs to sax
     * @param includeSubPrograms true to include subprograms
     * @throws MalformedURLException if an error occurred
     * @throws IOException if an error occurred
     * @throws SAXException if an error occurred
     */
    protected void _saxPrograms(AmetysObjectIterable<Program> programs, boolean includeSubPrograms) throws MalformedURLException, IOException, SAXException
    {
        Map<String, ContentAttributeDefinition> tableRefAttributeDefs = _odfTableRefHelper.getTableRefAttributeDefinitions(ProgramFactory.PROGRAM_CONTENT_TYPE);
        
        for (AbstractProgram program : programs)
        {
            _saxAbstractProgram("program", program, tableRefAttributeDefs, includeSubPrograms);
        }
    }
    
    /**
     * SAX a program or subprogram
     * @param tagName the XML root tag
     * @param program The abstract program to sax
     * @param tableRefAttributeDefs The table reference attribute definitions
     * @param includeSubprograms true to include subprograms
     * @throws MalformedURLException if an error occurred
     * @throws IOException if an error occurred
     * @throws SAXException if an error occurred
     */
    protected void _saxAbstractProgram(String tagName, AbstractProgram program, Map<String, ContentAttributeDefinition> tableRefAttributeDefs, boolean includeSubprograms) throws MalformedURLException, IOException, SAXException
    {
        try
        {
            _odfHelper.switchToLiveVersionIfNeeded(program);
            SitemapSource src = null;      
            
            try
            {
                AttributesImpl attrs = new AttributesImpl();
                attrs.addCDATAAttribute("id", program.getId());
                attrs.addCDATAAttribute("name", program.getName());
                attrs.addCDATAAttribute("title", program.getTitle());
                
                XMLUtils.startElement(contentHandler, tagName, attrs);
                
                _saxTableRefAttributeValues(program, tableRefAttributeDefs);
                
                XMLUtils.startElement(contentHandler, "fo");
                
                String uri = "cocoon://_plugins/odf/_content/" + program.getName() + ".fo";
                src = (SitemapSource) resolver.resolveURI(uri, null, Map.of("exportMode", __EXPORT_MODE));
                src.toSAX(new IgnoreRootHandler(contentHandler));
                
                XMLUtils.endElement(contentHandler, "fo");
                
                if (includeSubprograms)
                {
                    Map<String, ContentAttributeDefinition> subProgramTableRefAttributeDefs = _odfTableRefHelper.getTableRefAttributeDefinitions(SubProgramFactory.SUBPROGRAM_CONTENT_TYPE);
                    List<ProgramPart> programPartChildren = program.getProgramPartChildren();
                    for (ProgramPart programPart : programPartChildren)
                    {
                        if (programPart instanceof SubProgram)
                        {
                            _saxAbstractProgram("subprogram", (SubProgram) programPart, subProgramTableRefAttributeDefs, includeSubprograms);
                        }
                    }
                }
                XMLUtils.endElement(contentHandler, tagName);
                
            }
            catch (UnknownAmetysObjectException e)
            {
                // The content may be archived
            }
            finally
            {
                resolver.release(src);
            }
            
        }
        catch (NoLiveVersionException e)
        {
            getLogger().info("No live version found for program item " + program.getTitle() + " (" + program.getCode() + "). The program item will not appear in the PDF export.", e);
        }
    }
    
    /**
     * SAX enumerated values of an attribute 
     * @param program The program
     * @param tableRefAttributeDefs The table reference attribute definitions
     * @throws AmetysRepositoryException if an error occurred
     * @throws SAXException if an error occurred
     * @throws IOException if an error occurred
     */
    protected void _saxTableRefAttributeValues(AbstractProgram program, Map<String, ContentAttributeDefinition> tableRefAttributeDefs) throws AmetysRepositoryException, SAXException, IOException
    {
        // Build a view containing all the reference tables attributes
        View view = View.of(program.getModel(), tableRefAttributeDefs.keySet().toArray(new String[tableRefAttributeDefs.size()]));
        
        // Generate SAX events for the built view
        XMLUtils.startElement(contentHandler, "metadata");
        program.dataToSAX(contentHandler, view);
        XMLUtils.endElement(contentHandler, "metadata");
    }
}
