/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.tree;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.repository.Content;
import org.ametys.plugins.workflow.repository.WorkflowAwareAmetysObject;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.plugins.workflow.support.WorkflowProvider.AmetysObjectWorkflow;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.workflow.loader.StepDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * {@link ODFTreeIndicator} to display the workflow step
 *
 */
public class WorkflowStepIndicator extends AbstractStaticODFTreeIndicator 
{
    private WorkflowProvider _workflowProvider;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _workflowProvider = (WorkflowProvider) smanager.lookup(WorkflowProvider.ROLE);
    }
    
    public IndicatorData getIndicatorData(Content content)
    {
        if (content instanceof WorkflowAwareAmetysObject waContent)
        {
            long workflowId = waContent.getWorkflowId();
            int currentStepId = Math.toIntExact(waContent.getCurrentStepId());
        
            AmetysObjectWorkflow workflow = _workflowProvider.getAmetysObjectWorkflow(waContent);
        
            String workflowName = workflow.getWorkflowName(workflowId);
            WorkflowDescriptor workflowDescriptor = workflow.getWorkflowDescriptor(workflowName);
        
            if (workflowDescriptor != null)
            {
                StepDescriptor stepDescriptor = workflowDescriptor.getStep(currentStepId);
                if (stepDescriptor != null)
                {
                    return getIndicatorData(content, currentStepId, stepDescriptor);
                }
            }
        }
        
        return null;
    }
    
    /**
     * Get indicator for the given workflow step
     * @param content the content
     * @param stepId the step id
     * @param stepDescriptor the step descriptor
     * @return the indicator data
     */
    protected IndicatorData getIndicatorData(Content content, int stepId, StepDescriptor stepDescriptor)
    {
        I18nizableText workflowStepName = new I18nizableText("application", stepDescriptor.getName());
        
        switch (stepId)
        {
            case 3:
                return new IndicatorData(workflowStepName, null, "green-color ametysicon-sign-raw-check", Map.of());
            case 2:
                return new IndicatorData(workflowStepName, null, "orange-color ametysicon-body-part-finger", Map.of());
            default:
                return new IndicatorData(workflowStepName, null, "orange-color ametysicon-edit45", Map.of());
        }
    }
}
