/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf;

import java.util.Collections;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.repository.Content;
import org.ametys.cms.transformation.ConsistencyChecker.CHECK;
import org.ametys.cms.transformation.URIResolver;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * {@link URIResolver} for a Content.
 */
public class OdfURIResolver extends AbstractLogEnabled implements URIResolver, Serviceable
{
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public String resolve(String uri, boolean download, boolean absolute, boolean internal)
    {
        String contentId = uri;
        int i = uri.indexOf(";");
        if (i != -1)
        {
            contentId = uri.substring(0, i);
        }
                
        return "javascript:parent.Ametys.tool.ToolsManager.openTool('uitool-content', {id:'" + contentId + "'});";
    }
    
    @Override
    public String resolveImage(String uri, int height, int width, boolean download, boolean absolute, boolean internal)
    {
        return resolve(uri, download, absolute, internal);
    }
    
    @Override
    public String resolveImageAsBase64(String uri, int height, int width)
    {
        return resolve(uri, false, false, false);
    }
    
    @Override
    public String resolveBoundedImage(String uri, int maxHeight, int maxWidth, boolean download, boolean absolute, boolean internal)
    {
        return resolve(uri, download, absolute, internal);
    }
    
    @Override
    public String resolveBoundedImageAsBase64(String uri, int maxHeight, int maxWidth)
    {
        return resolve(uri, false, false, false);
    }
    
    @Override
    public String resolveCroppedImage(String uri, int cropHeight, int cropWidth, boolean download, boolean absolute, boolean internal)
    {
        return resolve(uri, download, absolute, internal);
    }
    
    @Override
    public String resolveCroppedImageAsBase64(String uri, int cropHeight, int cropWidth)
    {
        return resolve(uri, false, false, false);
    }
    
    @Override
    public CHECK checkLink(String uri, boolean shortTest)
    {
        try
        {
            _resolver.resolveById(uri);
            return CHECK.SUCCESS;
        }
        catch (UnknownAmetysObjectException e)
        {
            return CHECK.UNKNOWN;
        }
    }
    
    @Override
    public String getType()
    {
        return "odf";
    }
    
    @Override
    public I18nizableText getLabel(String uri)
    {
        try
        {
            Content content = _resolver.resolveById(uri);
            return new I18nizableText("plugin.odf", "PLUGINS_ODF_LINK_CONTENT_LABEL", Collections.singletonList(content.getTitle()));
        }
        catch (UnknownAmetysObjectException e)
        {
            return new I18nizableText("plugin.odf", "PLUGINS_ODF_LINK_CONTENT_UNKNOWN");
        }
    }
}
