/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.lheo;

import java.io.IOException;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.Response;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.xml.sax.SAXException;

import org.ametys.odf.program.AbstractProgram;

/**
 * Generates catalog to LHEO
 */
public class GlobalExportToLHEOGenerator extends ServiceableGenerator
{
    /** The export LHEO manager */
    protected ExportToLHEOManager _exportLHEOManager;
    
    /** The LHEO query manager */
    protected LHEOQueryManager _lheoQueryManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _exportLHEOManager = (ExportToLHEOManager) smanager.lookup(ExportToLHEOManager.ROLE);
        _lheoQueryManager = (LHEOQueryManager) smanager.lookup(LHEOQueryManager.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);

        String catalog = request.getParameter("catalog");
        String lang = request.getParameter("lang");
        
        List<AbstractProgram> programList = _lheoQueryManager.getAbstractProgramsToExport(catalog, lang, Map.of());
        
        // Configure the response header
        Response response = ObjectModelHelper.getResponse(objectModel);
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMddhhmmss");
        response.setHeader("Content-Disposition", "attachment; filename=\"LHEO_" + catalog + "_" + LocalDateTime.now().format(formatter) + ".xml" + "\"");
        
        _exportLHEOManager.saxLHEO(contentHandler, programList);
    }
}
