/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.migration;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.cms.CmsConstants;
import org.ametys.cms.data.ContentDataHelper;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.ModifiableDefaultContent;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.cms.workflow.CreateContentFunction;
import org.ametys.odf.course.Course;
import org.ametys.odf.course.CourseFactory;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.coursepart.CoursePartFactory;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.MetadataExpression;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.plugins.workflow.support.WorkflowProvider.AmetysObjectWorkflow;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.InvalidActionException;
import com.opensymphony.workflow.WorkflowException;

/**
 * Component class to migrate totalDurationOf* metadata.
 */
public class MigrateCoursePartComponent extends AbstractLogEnabled implements Serviceable, Component
{
    /** The component role. */
    public static final String ROLE = MigrateCoursePartComponent.class.getName();
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The ODF Reference table helper */
    protected OdfReferenceTableHelper _odfRefTableHelper;
    /** The content workflow helper */
    protected ContentWorkflowHelper _workflowHelper;
    /** The workflow */
    protected WorkflowProvider _workflowProvider;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _odfRefTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
        _workflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
        _workflowProvider = (WorkflowProvider) manager.lookup(WorkflowProvider.ROLE);
    }

    /**
     * Migrate the totalDurationOf to course parts.
     * @throws AmetysRepositoryException if an error occurs
     * @throws WorkflowException if an error occurs
     * @throws RepositoryException if an error occurs
     */
    public void migrateCourseParts() throws AmetysRepositoryException, WorkflowException, RepositoryException
    {
        migrateCourseParts(null, null);
    }
    
    /**
     * Migrate the totalDurationOf to course parts.
     * @param natureEnseignementCategories the map of nature enseignement categories
     * @param natureEnseignements the map of nature enseignement
     * @throws AmetysRepositoryException if an error occurs
     * @throws WorkflowException if an error occurs
     * @throws RepositoryException if an error occurs
     */
    public void migrateCourseParts(Map<String, Pair<String, Long>> natureEnseignementCategories, Map<String, Pair<String, String>> natureEnseignements) throws AmetysRepositoryException, WorkflowException, RepositoryException
    {
        Map<String, OdfReferenceTableEntry> natureByCode = _createNaturesEnseignement(natureEnseignementCategories, natureEnseignements);
        
        // Migrate existing totalDurationOf*
        Expression[] metadataExpressions =  natureByCode.keySet()
            .stream()
            .map(code -> new MetadataExpression("totalDurationOf" + code))
            .toArray(MetadataExpression[]::new);
        
        String xpathQuery = ContentQueryHelper.getContentXPathQuery(
            new AndExpression(
                new ContentTypeExpression(Operator.EQ, CourseFactory.COURSE_CONTENT_TYPE),
                new OrExpression(metadataExpressions)
            ));
        
        AmetysObjectIterable<Course> courses = _resolver.query(xpathQuery);
        for (Course course : courses)
        {
            boolean hasLiveVersion = ArrayUtils.contains(course.getAllLabels(), CmsConstants.LIVE_LABEL);
            boolean currentVersionIsLive = hasLiveVersion && ArrayUtils.contains(course.getLabels(), CmsConstants.LIVE_LABEL);
            
            if (hasLiveVersion && !currentVersionIsLive)
            {
                String currentVersion = course.getNode()
                    .getSession()
                    .getWorkspace()
                    .getVersionManager()
                    .getBaseVersion(course.getNode().getPath())
                    .getName();
                
                // switching to old live version
                course.restoreFromLabel(CmsConstants.LIVE_LABEL);
                
                // migrate old live version
                Map<String, CoursePart> createdCourseParts = _migrateCoursePart(course, natureByCode, true);
                
                // restore current version
                course.restoreFromRevision(currentVersion);
                
                // update current version
                _updateCoursePart(course, natureByCode, createdCourseParts);
            }
            else
            {
                // migrate current version
                _migrateCoursePart(course, natureByCode, currentVersionIsLive);
            }
        }
    }
    
    /**
     * Update the {@link CoursePart} and the course parts list of the {@link Course} with the values of the current version.
     * @param course The {@link Course} to migrate
     * @param natureByCode The course part natures
     * @param createdCourseParts The {@link Map} of the created {@link CoursePart} previously
     */
    protected void _updateCoursePart(Course course, Map<String, OdfReferenceTableEntry> natureByCode, Map<String, CoursePart> createdCourseParts)
    {
        JCRRepositoryData courseRepoData = new JCRRepositoryData(course.getNode());
        
        Set<String> courseParts = new HashSet<>();
        
        for (String natureCode : natureByCode.keySet())
        {
            String dataName = "totalDurationOf" + natureCode;
            double totalDurationOf = courseRepoData.hasValue(dataName) ? courseRepoData.getDouble(dataName) : 0;
            
            if (totalDurationOf > 0)
            {
                CoursePart coursePart = createdCourseParts.get(natureCode);
                if (coursePart == null)
                {
                    // Create course part
                    Optional.ofNullable(_createCoursePart(course, natureByCode.get(natureCode), totalDurationOf, false))
                        .ifPresent(createdCoursePart -> courseParts.add(createdCoursePart.getId()));
                }
                else
                {
                    // Update course part
                    double oldValue = coursePart.getValue(CoursePart.NB_HOURS, false, 0D);
                    if (oldValue != totalDurationOf)
                    {
                        coursePart.setValue(CoursePart.NB_HOURS, totalDurationOf);
                        coursePart.saveChanges();
                        coursePart.checkpoint();
                    }
                    
                    courseParts.add(coursePart.getId());
                }
            }
            
            removeExternalizableData(courseRepoData, dataName);
        }
        
        // Add the course part to the course
        if (!courseParts.isEmpty())
        {
            courseParts.addAll(ContentDataHelper.getContentIdsListFromMultipleContentData(course, Course.CHILD_COURSE_PARTS));
            course.setValue(Course.CHILD_COURSE_PARTS, courseParts.toArray(new String[courseParts.size()]));
        }
        else
        {
            course.removeValue(Course.CHILD_COURSE_PARTS);
            course.removeExternalizableMetadataIfExists(Course.CHILD_COURSE_PARTS);
        }
        
        if (course.needsSave())
        {
            course.saveChanges();
            course.checkpoint();
        }
    }
    
    /**
     * Create a course part linked to the course.
     * @param course The {@link Course} to migrate
     * @param natureByCode The course part natures
     * @param isLive Set the Live label if <code>true</code>
     * @return The created course parts
     */
    protected Map<String, CoursePart> _migrateCoursePart(Course course, Map<String, OdfReferenceTableEntry> natureByCode, boolean isLive)
    {
        JCRRepositoryData courseRepoData = new JCRRepositoryData(course.getNode());
        
        Map<String, CoursePart> createdCourseParts = new HashMap<>();
        
        // Create the course parts
        for (String natureCode : natureByCode.keySet())
        {
            String dataName = "totalDurationOf" + natureCode;
            double totalDurationOf = courseRepoData.hasValue(dataName) ? courseRepoData.getDouble(dataName) : 0;
            
            if (totalDurationOf > 0)
            {
                Optional.ofNullable(_createCoursePart(course, natureByCode.get(natureCode), totalDurationOf, isLive))
                    .ifPresent(coursePart -> createdCourseParts.put(natureCode, coursePart));
            }

            removeExternalizableData(courseRepoData, dataName);
        }
        
        // Add the course part to the course
        if (!createdCourseParts.isEmpty())
        {
            Set<String> coursePartIds = createdCourseParts.values()
                .stream()
                .map(CoursePart::getId)
                .collect(Collectors.toSet());
            
            coursePartIds.addAll(ContentDataHelper.getContentIdsListFromMultipleContentData(course, Course.CHILD_COURSE_PARTS));
            course.setValue(Course.CHILD_COURSE_PARTS, coursePartIds.toArray(new String[coursePartIds.size()]));
        }
        
        if (course.needsSave())
        {
            // Versions précédentes incompatibles
            course.addLabel("NotCompatible", true);
            
            // Sauvegarde et avancement
            course.saveChanges();
            course.checkpoint();
        }
        
        if (isLive)
        {
            course.addLabel(CmsConstants.LIVE_LABEL, true);
        }
        
        return createdCourseParts;
    }
    
    /**
     * Remove a data and its externalizable data
     * @param repositoryData the repository data containing the data
     * @param dataName name of the data to remove
     */
    protected void removeExternalizableData(JCRRepositoryData repositoryData, String dataName)
    {
        if (repositoryData.hasValue(dataName))
        {
            repositoryData.removeValue(dataName);
        }
        
        if (repositoryData.hasValue(dataName + ModelAwareDataHolder.ALTERNATIVE_SUFFIX))
        {
            repositoryData.removeValue(dataName + ModelAwareDataHolder.ALTERNATIVE_SUFFIX);
        }
        
        if (repositoryData.hasValue(dataName + ModelAwareDataHolder.STATUS_SUFFIX))
        {
            repositoryData.removeValue(dataName + ModelAwareDataHolder.STATUS_SUFFIX);
        }
    }
    
    /**
     * Create a course part linked to the course.
     * @param course The {@link Course} holder
     * @param nature The nature of the course part
     * @param totalDurationOf The number of hours
     * @param isLive Set the Live label if <code>true</code>
     * @return The {@link CoursePart} id
     */
    protected CoursePart _createCoursePart(Course course, OdfReferenceTableEntry nature, Double totalDurationOf, boolean isLive)
    {
     // Create the course part
        String coursePartTitle = course.getTitle();
        if (nature != null)
        {
            coursePartTitle += " - " + nature.getCode();
        }
        String coursePartName = NameHelper.filterName(coursePartTitle);
        
        Map<String, Object> resultMap = new HashMap<>();
        
        Map<String, Object> inputs = new HashMap<>();
        inputs.put(CreateContentFunction.CONTENT_LANGUAGE_KEY, course.getLanguage());
        inputs.put(CreateContentFunction.CONTENT_NAME_KEY, coursePartName); 
        inputs.put(CreateContentFunction.CONTENT_TITLE_KEY, coursePartTitle);
        inputs.put(CreateContentFunction.CONTENT_TYPES_KEY, new String[] {CoursePartFactory.COURSE_PART_CONTENT_TYPE});
        
        Map<String, Object> results = new HashMap<>();
        inputs.put(AbstractWorkflowComponent.RESULT_MAP_KEY, results);
        
        CoursePart coursePart = null;
        try
        {
            Map<String, Object> workflowResult = _workflowHelper.createContent("course-part", 1, coursePartName, coursePartTitle, new String[] {CoursePartFactory.COURSE_PART_CONTENT_TYPE}, null, course.getLanguage());
            coursePart = (CoursePart) workflowResult.get(AbstractContentWorkflowComponent.CONTENT_KEY);
            
            if (nature != null)
            {
                coursePart.setValue(CoursePart.NATURE, nature.getId());
            }
            coursePart.setValue(CoursePart.NB_HOURS, totalDurationOf);
            coursePart.setValue(CoursePart.COURSE_HOLDER, course.getId());
            coursePart.setValue(CoursePart.PARENT_COURSES, new String[] {course.getId()});
            coursePart.setValue(CoursePart.CATALOG, course.getCatalog());
            
            _setAdditionalValues(coursePart, course);
            
            coursePart.saveChanges();
            coursePart.checkpoint();
            
            if (isLive)
            {
                coursePart.addLabel(CmsConstants.LIVE_LABEL, true);
            }
        }
        catch (WorkflowException e)
        {
            resultMap.put("error", Boolean.TRUE);
            getLogger().error("Failed to initialize workflow for content '{}' and language '{}'", coursePartTitle, course.getLanguage(), e);
        }
        
        return coursePart;
    }
    
    /**
     * Set additional values to the {@link CoursePart} from the {@link Course}.
     * @param coursePart The course part to modify
     * @param course The original course
     */
    protected void _setAdditionalValues(CoursePart coursePart, Course course)
    {
        // Nothing to do
    }
    
    /**
     * Create entries into the reference table EnseignementNature if they don't exist.
     * @param natureEnseignementCategories the map of nature enseignement categories
     * @param natureEnseignements the map of nature enseignement
     * @return The list of natures with their code and associated ID.
     * @throws AmetysRepositoryException if an error occurs
     * @throws WorkflowException if an error occurs
     */
    protected Map<String, OdfReferenceTableEntry> _createNaturesEnseignement(Map<String, Pair<String, Long>> natureEnseignementCategories, Map<String, Pair<String, String>> natureEnseignements) throws AmetysRepositoryException, WorkflowException
    {
        Map<String, String> categoryByCode = _createNatureEnseignementCategories(natureEnseignementCategories);
        Map<String, OdfReferenceTableEntry> natureByCode = new HashMap<>();
        
        for (Map.Entry<String, Pair<String, String>> nature : _getNaturesEnseignementList(natureEnseignements).entrySet())
        {
            String natureCode = nature.getKey();
            OdfReferenceTableEntry natureEntry = _odfRefTableHelper.getItemFromCode(OdfReferenceTableHelper.ENSEIGNEMENT_NATURE, natureCode);
            
            if (natureEntry == null)
            {
                String title = nature.getValue().getLeft();
                String categoryCode = nature.getValue().getRight();
                Map<String, String> titleVariants = new HashMap<>();
                titleVariants.put("fr", title);
                Map<String, Object> result = _workflowHelper.createContent("reference-table", 1, title, titleVariants, new String[] {OdfReferenceTableHelper.ENSEIGNEMENT_NATURE}, new String[0]);
                ModifiableDefaultContent natureContent = (ModifiableDefaultContent) result.get(AbstractContentWorkflowComponent.CONTENT_KEY);
                natureContent.setValue(OdfReferenceTableEntry.CODE, natureCode);
                natureContent.setValue("category", categoryByCode.get(categoryCode));
                natureContent.saveChanges();
                _doAction(natureContent, 22);
                
                natureEntry = new OdfReferenceTableEntry(natureContent);
            }
            
            natureByCode.put(natureCode, natureEntry);
        }
        
        return natureByCode;
    }

    /**
     * Create entries into the reference table EnseignementNature if they don't exist.
     * @param natureEnseignementCategories the map of nature enseignement categories
     * @return The list of natures with their code and associated ID.
     * @throws AmetysRepositoryException if an error occurs
     * @throws WorkflowException if an error occurs
     */
    protected Map<String, String> _createNatureEnseignementCategories(Map<String, Pair<String, Long>> natureEnseignementCategories) throws AmetysRepositoryException, WorkflowException
    {
        Map<String, String> categoryByCode = new HashMap<>();
        
        for (Map.Entry<String, Pair<String, Long>> category : _getNaturesEnseignementCategoryList(natureEnseignementCategories).entrySet())
        {
            String categoryCode = category.getKey();
            OdfReferenceTableEntry categoryEntry = _getOrCreateNatureEnseignement(category.getValue().getLeft(), categoryCode, category.getValue().getRight());
            categoryByCode.put(categoryCode, categoryEntry.getId());
        }
        
        return categoryByCode;
    }
    
    /**
     * List of the course parts natures.
     * @param natureEnseignements the map of nature enseignement
     * @return A {@link Map} with the code as a key, and a {@link Pair} with the title and the category code as a value
     */
    protected Map<String, Pair<String, String>> _getNaturesEnseignementList(Map<String, Pair<String, String>> natureEnseignements)
    {
        if (natureEnseignements != null && !natureEnseignements.isEmpty())
        {
            return natureEnseignements;
        }
        
        Map<String, Pair<String, String>> natures = new HashMap<>();
        natures.put("CM", Pair.of("Cours Magistral", "CM"));
        natures.put("TD", Pair.of("Travaux Dirigés", "TD"));
        natures.put("TP", Pair.of("Travaux Pratique", "TP"));
        return natures;
    }
    
    /**
     * List of the course parts nature categories.
     * @param natureEnseignementCategories the map of nature enseignement categories
     * @return A {@link Map} with the code as a key, and the title as a value
     */
    protected Map<String, Pair<String, Long>> _getNaturesEnseignementCategoryList(Map<String, Pair<String, Long>> natureEnseignementCategories)
    {
        if (natureEnseignementCategories != null && !natureEnseignementCategories.isEmpty())
        {
            return natureEnseignementCategories;
        }
        
        Map<String, Pair<String, Long>> categories = new HashMap<>();
        categories.put("CM", Pair.of("Cours Magistral", 1L));
        categories.put("TD", Pair.of("Travaux Dirigés", 2L));
        categories.put("TP", Pair.of("Travaux Pratique", 3L));
        return categories;
    }
    
    /**
     * {@link ContentWorkflowHelper} cannot be used in these conditions.
     * @param content The content
     * @param actionId Action to perform
     * @return The result map
     * @throws WorkflowException if an error occurs
     * @throws AmetysRepositoryException if an error occurs
     */
    protected Map<String, Object> _doAction(WorkflowAwareContent content, Integer actionId) throws AmetysRepositoryException, WorkflowException
    {
        Map<String, Object> inputs = new HashMap<>();
        Map<String, Object> results = new HashMap<>();
        inputs.put(AbstractWorkflowComponent.RESULT_MAP_KEY, results);
        inputs.put(AbstractContentWorkflowComponent.CONTENT_KEY, content);
        inputs.put(AbstractWorkflowComponent.FAIL_CONDITIONS_KEY, new ArrayList<>());
        inputs.put(AbstractWorkflowComponent.CONTEXT_PARAMETERS_KEY, new HashMap<>());
        
        try
        {
            AmetysObjectWorkflow workflow = _workflowProvider.getAmetysObjectWorkflow(content);
            workflow.doAction(content.getWorkflowId(), actionId, inputs);
        }
        catch (InvalidActionException e)
        {
            getLogger().error("An error occured while do workflow action '{}' on content '{}'", actionId, content.getId(), e);
            throw e; 
        }
        
        return results;
    }
    
    /**
     * Get or create the nature enseignement if it doesn't exist. The code is tested.
     * @param title The title of the nature
     * @param code The code of the nature
     * @param order The order to set.
     * @return The corresponding entry
     * @throws AmetysRepositoryException if an error occurs
     * @throws WorkflowException if an error occurs
     */
    protected OdfReferenceTableEntry _getOrCreateNatureEnseignement(String title, String code, Long order) throws AmetysRepositoryException, WorkflowException
    {
        OdfReferenceTableEntry categoryEntry = _odfRefTableHelper.getItemFromCode(OdfReferenceTableHelper.ENSEIGNEMENT_NATURE_CATEGORY, code);
        
        if (categoryEntry == null)
        {
            Map<String, Object> result = _workflowHelper.createContent("reference-table", 1, title, title, new String[] {OdfReferenceTableHelper.ENSEIGNEMENT_NATURE_CATEGORY}, new String[0], "fr");
            ModifiableDefaultContent categoryContent = (ModifiableDefaultContent) result.get(AbstractContentWorkflowComponent.CONTENT_KEY);
            categoryContent.setValue(OdfReferenceTableEntry.CODE, code);
            categoryContent.setValue("order", order);
            categoryContent.saveChanges();
            _doAction(categoryContent, 22);
            
            categoryEntry = new OdfReferenceTableEntry(categoryContent);
        }
        
        return categoryEntry;
    }
}
