/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.observation.skill;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;

/**
 * Observer to delete skills that became orphans after being unlinked from their parent.
 * We compute the skills to delete by comparing the previous skills with the current ones.
 * When the modified content is a Program, delete the linked macro skills that became orphans
 *    The deletion of macro skills will trigger the deletion of its micro skills
 * When the modified content is a Macro skill, delete the linked micro skills that became orphans
 */
public class UpdateContentSkillStep2Observer extends AbstractSkillsStepObserver
{
    @Override
    protected String getSupportedEventId()
    {
        return ObservationConstants.EVENT_CONTENT_MODIFIED;
    }
    
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Content content = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        // When the content is modified, we need to check if the skills are still linked to the content
        ContentValue[] previousSkills = _getRequestAttribute("previousSkills", content.getId());
        if (previousSkills != null)
        {
            ContentValue[] currentSkills = _getSkills(content);

            // Compare the previous skills with the current ones to retrieve the skills which became orphans to delete them
            ContentValue[] skillsToDeleteContentValue = getSkillsToDelete(previousSkills, currentSkills);
            if (skillsToDeleteContentValue != null)
            {
                List<Content> skillsToDelete = Arrays.stream(skillsToDeleteContentValue)
                        .map(ContentValue::getContentIfExists)
                        .filter(Optional::isPresent)
                        .map(Optional::get)
                        .map(Content.class::cast)
                        .toList();
                
                if (!skillsToDelete.isEmpty())
                {
                    // Remove the orphan skills
                    _contentDAO.forceDeleteContentsObj(skillsToDelete, null);
                }
            }
        }
    }
}
