/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.rncp;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.TraversableSource;
import org.docx4j.openpackaging.exceptions.Docx4JException;
import org.docx4j.openpackaging.packages.WordprocessingMLPackage;
import org.docx4j.openpackaging.parts.DocPropsCustomPart;
import org.xml.sax.SAXException;

import org.ametys.core.util.URIUtils;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.google.common.collect.ImmutableMap;

/**
 * Helper for RNCP models.
 */
public class RNCPModelHelper extends AbstractLogEnabled implements Component, Serviceable, Initializable
{
    /** The component role */
    public static final String ROLE = RNCPModelHelper.class.getName();

    private static final String RNCP_DIR = "context://WEB-INF/param/rncp";

    /** The source resolver */
    protected SourceResolver _srcResolver;
    
    /** URL to update DOCX properties */
    private String _baseURL;

    @Override
    public void initialize() throws Exception
    {
        _baseURL = Config.getInstance().getValue("cms.url") + "/_odf/request/get-metadata-value.xml?contentId=#{CONTENTID}&metadata=";
    }
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _srcResolver = (SourceResolver) smanager.lookup(SourceResolver.ROLE);
    }
    
    /**
     * Get the list of RNCP models.
     * @return A JSON list with id and label of each available RNCP model
     * @throws ConfigurationException if an error occurs
     * @throws SAXException if an error occurs
     * @throws IOException if an error occurs
     */
    public List<Map<String, Object>> getModels() throws ConfigurationException, SAXException, IOException
    {
        List<Map<String, Object>> models = new ArrayList<>();

        TraversableSource rncpDir = (TraversableSource) _srcResolver.resolveURI(RNCP_DIR);
        if (rncpDir.exists() && rncpDir.isCollection())
        {
            for (TraversableSource file : (Collection<TraversableSource>) rncpDir.getChildren())
            {
                String fileName = file.getName();
                if (!file.isCollection() && fileName.endsWith(".docx"))
                {
                    models.add(ImmutableMap.of("id", fileName, "label", fileName));
                }
            }
        }
        else
        {
            getLogger().warn("There is no directory '{}'.", RNCP_DIR);
        }
        
        return models;
    }
    
    /**
     * Update the RNCP model with the good properties.
     * @param modelId The model ID, shouldn't be null
     * @param programId The mention ID
     * @param subProgramId The subprogram ID
     * @param out The {@link OutputStream}
     * @throws MalformedURLException if an error occurs
     * @throws IOException if an error occurs
     */
    public void updateAbstractProgramRNCPSheet(String modelId, String programId, String subProgramId, OutputStream out) throws MalformedURLException, IOException
    {
        TraversableSource rncpDir = (TraversableSource) _srcResolver.resolveURI(RNCP_DIR);
        Source childSrc = rncpDir.getChild(modelId);
        if (!childSrc.exists())
        {
            throw new IllegalArgumentException("The model ID is not valid.");
        }
        try (InputStream is = childSrc.getInputStream())
        {
            WordprocessingMLPackage wordMLPackage = WordprocessingMLPackage.load(is);
            
            DocPropsCustomPart docPropsCustomPart = wordMLPackage.getDocPropsCustomPart();
            docPropsCustomPart.setProperty("ParcoursURL", _baseURL.replace("#{CONTENTID}", URIUtils.encodeParameter(subProgramId)));
            docPropsCustomPart.setProperty("MentionURL", _baseURL.replace("#{CONTENTID}", URIUtils.encodeParameter(programId)));
            
            wordMLPackage.save(out);
        }
        catch (Docx4JException e)
        {
            throw new IOException("Error while modifying the document '" + modelId + "'.", e);
        }
    }
}


