/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.tree;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.repository.Content;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * {@link ODFTreeIndicator} for publishable status
 *
 */
public class NotPublishableIndicator extends AbstractStaticODFTreeIndicator
{
    private ODFHelper _odfHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
    }
    
    public IndicatorData getIndicatorData(Content content)
    {
        if (content instanceof ProgramItem programItem)
        {
            if (!programItem.isPublishable())
            {
                return new IndicatorData(new I18nizableText("plugin.odf", "PLUGINS_ODF_CONTENTS_TREE_INDICATORS_NOT_PUBLISHABLE_TOOLTIP"), null, "ametysicon-eye-blocked", Map.of());
            }
            else if (!_isParentPublishable(programItem))
            {
                return new IndicatorData(new I18nizableText("plugin.odf", "PLUGINS_ODF_CONTENTS_TREE_INDICATORS_PARENT_NOT_PUBLISHABLE_TOOLTIP"), null, "grey-color ametysicon-eye-blocked", Map.of());
            }
        }
        
        return null;
    }
    
    private boolean _isParentPublishable(ProgramItem content)
    {
        List<ProgramItem> parents = _odfHelper.getParentProgramItems(content);
        if (parents.isEmpty())
        {
            return true;
        }
        
        return parents.stream()
                .filter(ProgramItem::isPublishable)
                .filter(this::_isParentPublishable)
                .findAny()
                .isPresent();
    }

}
