/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.tree;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.repository.Content;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.program.Program;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;

/**
 * {@link ODFTreeIndicator} to display the share status
 */
public class ShareStatusIndicator extends AbstractStaticODFTreeIndicator
{
    private ODFHelper _odfHelper;
    private AmetysObjectResolver _resolver;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public IndicatorData getIndicatorData(Content content)
    {
        if (content instanceof ProgramItem programItem)
        {
            int nbParents = _odfHelper.getParentProgramItems(programItem).size();
            boolean isShared = nbParents > 1;
            boolean isSharedByParents = !isShared ? _isSharedByParents(programItem) : false;
            
            if (isShared)
            {
                long nbOrphanParents = _nbOrphanParents(programItem);
                if (nbOrphanParents > 0)
                {
                    // Program item is shared with orphan parents
                    Map<String, I18nizableTextParameter> i18nParams = Map.of("nbParents", new I18nizableText(String.valueOf(nbParents)), 
                            "nbOrphanParents", new I18nizableText(String.valueOf(nbOrphanParents)));
                    return new IndicatorData(new I18nizableText("plugin.odf", "PLUGINS_ODF_CONTENTS_TREE_INDICATORS_SHARED_WITH_ORPHANS", i18nParams), null, "odficon-share-filled orange-color", Map.of());
                }
                else
                {
                    Map<String, I18nizableTextParameter> i18nParams = Map.of("nbParents", new I18nizableText(String.valueOf(nbParents)));
                    return new IndicatorData(new I18nizableText("plugin.odf", "PLUGINS_ODF_CONTENTS_TREE_INDICATORS_SHARED", i18nParams), null, "odficon-share-filled", Map.of());
                }
            }
            else if (isSharedByParents)
            {
                return new IndicatorData(new I18nizableText("plugin.odf", "PLUGINS_ODF_CONTENTS_TREE_INDICATORS_PARENT_SHARED"), null, "odficon-share-outline", Map.of());
            }
        }
        
        return null;
    }
    
    private long _nbOrphanParents(ProgramItem programItem)
    {
        List<EducationalPath> educationalPaths = _odfHelper.getEducationalPaths(programItem);
        
        return educationalPaths.stream()
            .filter(p -> !(p.getProgramItems(_resolver).get(0) instanceof Program))
            .count();
    }
    
    private boolean _isShared(ProgramItem programItem)
    {
        return _odfHelper.getParentProgramItems(programItem).size() > 1;
    }
    
    private boolean _isSharedByParents(ProgramItem programItem)
    {
        for (ProgramItem parent : _odfHelper.getParentProgramItems(programItem))
        {
            return _isShared(programItem) ? true : _isSharedByParents(parent);
        }
        return false;
    }

}
